(function () {
  'use strict';

  // Remueve la clase 'preload' del body, que sirve para mostrar los elementos más rápido
  window.addEventListener('DOMContentLoaded', () => {
    const body = document.querySelector('body');
    body.classList.remove('preload');
  });
  const buttons = document.querySelectorAll('[data-outside]');
  const ACTIVE_CLASS = 'is-active';
  function outsideClick(button) {
    if (!button) return;
    const target = document.getElementById(button.dataset.outside);
    if (!target) return;
    function toggleClasses() {
      button.classList.toggle(ACTIVE_CLASS);
      target.classList.toggle(ACTIVE_CLASS);
      if (button.classList.contains(ACTIVE_CLASS)) {
        document.addEventListener('click', clickOutside);
        return;
      }
      document.removeEventListener('click', clickOutside);
    }
    button.addEventListener('click', toggleClasses);
    function clickOutside(event) {
      if (!target.contains(event.target) && !button.contains(event.target)) {
        toggleClasses();
        document.removeEventListener('click', clickOutside);
      }
    }
    const closeButton = target.querySelector('[data-close]');
    if (closeButton) {
      closeButton.addEventListener('click', () => {
        button.classList.remove(ACTIVE_CLASS);
        target.classList.remove(ACTIVE_CLASS);
        document.removeEventListener('click', clickOutside);
      });
    }
  }
  buttons.forEach(button => {
    outsideClick(button);
  });
  const tiltBox = document.querySelector('.tilt-box');
  const rect = tiltBox.getBoundingClientRect();
  const maxTilt = 12; // Cantidad maxima de inclinacion
  const extraMargin = 0; // Margen adicional en pixeles alrededor del cuadro

  const expandedRect = {
    left: rect.left - extraMargin,
    right: rect.right + extraMargin,
    top: rect.top - extraMargin,
    bottom: rect.bottom + extraMargin,
    width: rect.width + extraMargin * 2,
    height: rect.height + extraMargin * 2
  };

  // Listener dentro del área expandida
  document.addEventListener('mousemove', e => {
    const mouseX = e.clientX;
    const mouseY = e.clientY;

    // Verificamos si el cursor está dentro del área expandida
    if (mouseX >= expandedRect.left && mouseX <= expandedRect.right && mouseY >= expandedRect.top && mouseY <= expandedRect.bottom) {
      const x = mouseX - expandedRect.left;
      const y = mouseY - expandedRect.top;
      const halfWidth = expandedRect.width / 2;
      const halfHeight = expandedRect.height / 2;

      // Calculamos la inclinación basada en la posición del mouse en el área expandida
      const tiltX = (x - halfWidth) / halfWidth * maxTilt;
      const tiltY = -((y - halfHeight) / halfHeight) * maxTilt;

      // Aplicamos la transformación
      tiltBox.style.transform = `perspective(1000px) rotateX(${tiltY}deg) rotateY(${tiltX}deg)`;
    } else {
      // Reset de inclinación cuando el mouse sale del área expandida
      tiltBox.style.transform = '';
    }
  });
  const headerBar = document.querySelector('.header-bar');
  window.addEventListener('scroll', function () {
    if (window.scrollY > 0) {
      headerBar.classList.add('header-fixed');
    } else {
      headerBar.classList.remove('header-fixed');
    }
  });

  // Función para inicializar el lienzo (canvas)
  function initCanvas(container) {
    const canvas = document.createElement('canvas');
    canvas.setAttribute('id', 'visualizerCanvas');
    canvas.setAttribute('class', 'visualizer-item');
    container.appendChild(canvas);
    canvas.width = container.clientWidth;
    canvas.height = container.clientHeight;
    return canvas;
  }

  // Función para cambiar el lienzo según el tamaño del contenedor
  function resizeCanvas(canvas, container) {
    canvas.width = container.clientWidth;
    canvas.height = container.clientHeight;
  }

  // Visualizer
  const visualizer = (audio, container) => {
    if (!audio || !container) {
      return;
    }
    const options = {
      fftSize: container.dataset.fftSize || 2048,
      numBars: container.dataset.bars || 80,
      maxHeight: 255
    };
    const ctx = new AudioContext();
    const audioSource = ctx.createMediaElementSource(audio);
    const analyzer = ctx.createAnalyser();
    audioSource.connect(analyzer);
    audioSource.connect(ctx.destination);
    const frequencyData = new Uint8Array(analyzer.frequencyBinCount);
    const canvas = initCanvas(container);
    const canvasCtx = canvas.getContext('2d');

    // Crear barras
    const renderBars = () => {
      resizeCanvas(canvas, container);
      analyzer.getByteFrequencyData(frequencyData);
      if (options.fftSize) {
        analyzer.fftSize = options.fftSize;
      }
      canvasCtx.clearRect(0, 0, canvas.width, canvas.height);
      for (let i = 0; i < options.numBars; i++) {
        const index = Math.floor((i + 10) * (i < options.numBars / 2 ? 2 : 1));
        const fd = frequencyData[index];
        const barHeight = Math.max(4, fd / 255 * options.maxHeight);
        const barWidth = canvas.width / options.numBars;
        const x = i * barWidth;
        const y = canvas.height - barHeight;
        canvasCtx.fillStyle = 'white';
        canvasCtx.fillRect(x, y, barWidth - 2, barHeight);
      }
      requestAnimationFrame(renderBars);
    };
    renderBars();

    // Listener del cambio de espacio en la ventana
    window.addEventListener('resize', () => {
      resizeCanvas(canvas, container);
    });
  };

  /* eslint-disable camelcase */
  // Función para obtener el día actual en formato de 2 letras
  function getCurrentDay() {
    const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
    const date = new Date();
    const day = date.getDay();
    return days[day];
  }
  function compareNumbers(a, b) {
    return Number(a) === Number(b);
  }
  function sortByDay(programas) {
    const resultado = {
      monday: [],
      tuesday: [],
      wednesday: [],
      thursday: [],
      friday: [],
      saturday: [],
      sunday: []
    };
    programas.forEach(programa => {
      const {
        prog_horario_lu,
        prog_horario_ma,
        prog_horario_mi,
        prog_horario_ju,
        prog_horario_vi,
        prog_horario_sa,
        prog_horario_do,
        prog_lu,
        prog_ma,
        prog_mi,
        prog_ju,
        prog_vi,
        prog_sa,
        prog_do,
        prog_titulo,
        prog_descripcion,
        prog_foto
      } = programa;
      const item = {
        titulo: prog_titulo,
        descripcion: prog_descripcion,
        foto: prog_foto,
        horario: null
      };

      // Verificar cada día y agregar al arreglo correspondiente si está activo
      if (compareNumbers(prog_lu, 1)) {
        item.horario = prog_horario_lu;
        resultado.monday.push({
          ...item
        });
      }
      if (compareNumbers(prog_ma, 1)) {
        item.horario = prog_horario_ma;
        resultado.tuesday.push({
          ...item
        });
      }
      if (compareNumbers(prog_mi, 1)) {
        item.horario = prog_horario_mi;
        resultado.wednesday.push({
          ...item
        });
      }
      if (compareNumbers(prog_ju, 1)) {
        item.horario = prog_horario_ju;
        resultado.thursday.push({
          ...item
        });
      }
      if (compareNumbers(prog_vi, 1)) {
        item.horario = prog_horario_vi;
        resultado.friday.push({
          ...item
        });
      }
      if (compareNumbers(prog_sa, 1)) {
        item.horario = prog_horario_sa;
        resultado.saturday.push({
          ...item
        });
      }
      if (compareNumbers(prog_do, 1)) {
        item.horario = prog_horario_do;
        resultado.sunday.push({
          ...item
        });
      }
    });

    // Ordenar los programas por hora en cada día
    Object.keys(resultado).forEach(dia => {
      resultado[dia].sort((a, b) => a.horario.localeCompare(b.horario));
    });
    return resultado;
  }

  // Compara la hora actual con la hora de inicio de un programa
  // @param {string} time - Hora de inicio del programa
  // @returns {number} - 1 si la hora actual es mayor, -1 si la hora actual es menor, 0 si son iguales
  function compareTime(time) {
    const date = new Date();
    const currentHour = date.getHours();
    const currentMinutes = date.getMinutes();
    const [programHour, programMinutes] = time.split(':').map(Number);
    if (currentHour > programHour || currentHour === programHour && currentMinutes > programMinutes) {
      return 1; // Hora actual es mayor
    } else if (currentHour < programHour || currentHour === programHour && currentMinutes < programMinutes) {
      return -1; // Hora actual es menor
    } else {
      return 0; // Horas iguales
    }
  }
  function setupSchedule(data, baseUrl) {
    const programs = sortByDay(data);
    const days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
    const tabs = document.querySelectorAll('.program-button');
    const tabsElements = document.querySelectorAll('.nexo-schedule-list');
    tabs.forEach(tab => {
      tab.onclick = event => {
        const target = event.target;
        const tabId = target.dataset.tab;
        tabs.forEach(tab => {
          tab.classList.remove('is-active');
        });
        tabsElements.forEach(tab => {
          tab.classList.remove('is-active');
        });
        const ul = document.getElementById(tabId);
        ul.classList.add('is-active');
        target.classList.add('is-active');
      };
    });
    days.forEach(day => {
      const ul = document.getElementById(`program-${day}`);
      ul.innerHTML = '';
      programs[day].forEach(item => {
        const li = document.createElement('li');
        li.classList.add('nexo-schedule-item');
        li.innerHTML = `
        <div class="nexo-schedule-item-picture">
          <img src="${baseUrl}${item.foto}" alt="${item.titulo}">
        </div>
        <div class="nexo-schedule-item-content">
          <h3 class="nexo-schedule-item-title">${item.titulo}</h3>
          <p class="nexo-schedule-item-description">${item.horario} - ${item.descripcion}</p>
        </div>
      `;

        // Comparar la hora actual con la hora de inicio del programa
        const result = compareTime(item.horario);

        // Agregar la clase is-past si la hora actual es mayor
        // Solo agregar las clases si el programa es del día actual
        if (result > 0 && day === getCurrentDay()) {
          li.classList.add('is-past');
        }
        ul.appendChild(li);
      });

      // al ultimo elemento con la clase is-past agregarle la clase is-current
      // Solo si el día es el día actual
      const pastItems = document.querySelectorAll('.is-past');
      if (pastItems.length > 0 && day === getCurrentDay()) {
        pastItems[pastItems.length - 1].classList.add('is-current');
      }
    });

    // Mostrar el día actual
    const currentDay = getCurrentDay();

    // Mostrar el día actual
    const alltabs = document.querySelectorAll('.program-button');
    alltabs.forEach(tab => {
      tab.classList.remove('is-active');
    });
    const currentTab = document.querySelector(`[data-tab="program-${currentDay}"]`);
    currentTab.classList.add('is-active');
    const allLists = document.querySelectorAll('.nexo-schedule-list');
    allLists.forEach(list => {
      list.classList.remove('is-active');
    });
    const currentList = document.getElementById(`program-${currentDay}`);
    currentList.classList.add('is-active');
  }

  const icons = {
    facebook: '<i class="ri-facebook-circle-fill"></i>',
    messenger: '<i class="ri-messenger-fill"></i>',
    youtube: '<i class="ri-youtube-fill"></i>',
    instagram: '<i class="ri-instagram-fill"></i>',
    whatsapp: '<i class="ri-whatsapp-fill"></i>',
    tiktok: '<i class="ri-tiktok-fill"></i>',
    play: '<i class="ri-play-fill"></i>',
    pause: '<i class="ri-pause-fill"></i>',
    volumeDown: '<i class="ri-volume-down-fill"></i>',
    volumeMute: '<i class="ri-volume-mute-fill"></i>',
    volumeUp: '<i class="ri-volume-up-fill"></i>',
    spotify: '<i class="ri-spotify-fill"></i>',
    bluesky: '<i class="ri-bluesky-fill"></i>',
    discord: '<i class="ri-discord-fill"></i>',
    soundcloud: '<i class="ri-soundcloud-fill"></i>',
    threads: '<i class="ri-threads-fill"></i>',
    twitter: '<i class="ri-twitter-fill"></i>',
    x: '<i class="ri-twitter-x-fill"></i>'
  };
  const svgIcons = {
    play: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M10 16.5v-9l6 4.5M12 2A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>',
    pause: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M15 16h-2V8h2m-4 8H9V8h2m1-6A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>',
    facebook: '<svg class="i i-facebook" viewBox="0 0 24 24"><path d="M17 14h-3v8h-4v-8H7v-4h3V7a5 5 0 0 1 5-5h3v4h-3q-1 0-1 1v3h4Z"></path></svg>',
    youtube: '<svg class="i i-youtube" viewBox="0 0 24 24"><path d="M1.5 17q-1-5.5 0-10Q1.9 4.8 4 4.5q8-1 16 0 2.1.3 2.5 2.5 1 4.5 0 10-.4 2.2-2.5 2.5-8 1-16 0-2.1-.3-2.5-2.5Zm8-8.5v7l6-3.5Z"></path></svg>',
    instagram: '<svg class="i i-instagram" viewBox="0 0 24 24"><circle cx="12" cy="12" r="4"></circle><rect width="20" height="20" x="2" y="2" rx="5"></rect><path d="M17.5 6.5h0"></path></svg>',
    whatsapp: '<svg class="i i-whatsapp" viewBox="0 0 24 24"><circle cx="9" cy="9" r="1"></circle><circle cx="15" cy="15" r="1"></circle><path d="M8 9a7 7 0 0 0 7 7m-9 5.2A11 11 0 1 0 2.8 18L2 22Z"></path></svg>',
    tiktok: '<svg class="i i-tiktok" viewBox="0 0 24 24"><path d="M22 6v5q-4 0-6-2v7a7 7 0 1 1-5-6.7m0 6.7a2 2 0 1 0-2 2 2 2 0 0 0 2-2V1h5q2 5 6 5"></path></svg>',
    bluesky: '<svg class="i i-bluesky" viewBox="0 0 24 24"><path d="M12 10Q2-2 2 6t5 8q-5 3-1 6t6-3q2 6 6 3t-1-6q5 0 5-8t-10 4"></path></svg>',
    discord: '<svg class="i i-discord" viewBox="0 0 24 24"><path d="M9 3q-2.5.5-5 2-3 5-3 12 2 2.5 6 4 1-1.5 1.5-3.5M7 17q5 2 10 0m-1.5.5q.5 2 1.5 3.5 4-1.5 6-4 0-7-3-12-2.5-1.5-5-2l-1 2q-2-.5-4 0L9 3"></path><circle cx="8" cy="12" r="1"></circle><circle cx="16" cy="12" r="1"></circle></svg>',
    soundcloud: '<svg class="i i-soundcloud" viewBox="0 0 24 24"><path d="M11 7v11m-3 0v-7m-6 6v-3m3-4v8m9 0h6a1 1 0 0 0 0-6 6 6 0 0 0-6-6Z"></path></svg>',
    threads: '<svg class="i i-threads" viewBox="0 0 24 24"><path d="M20 8C18 0 6 0 4 8s1.5 14 8 14 10-7.6 4-10-9 2-7 4 7 1 7-4-5-6-7-4"></path></svg>',
    twitter: '<svg class="i i-twitter" viewBox="0 0 24 24"><path d="M12 7.5a4.5 4.5 0 0 1 8-3Q22 4 23 3q0 2-2 4A13 13 0 0 1 1 19q5 0 7-2-8-4-5-13 4 5 9 5Z"></path></svg>',
    x: '<svg class="i i-x" viewBox="0 0 24 24"><path d="m3 21 7.5-7.5m3-3L21 3M8 3H3l13 18h5Z"></path></svg>',
    deezer: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" d="M20.1 3.6c.2-1.2.6-2 1-2 .6 0 1.1 2.8 1.1 6.2 0 3.5-.5 6.3-1.2 6.3-.3 0-.5-.5-.7-1.3-.3 3-1 5-1.8 5-.6 0-1.1-1.3-1.5-3.2-.3 3.7-.9 6.2-1.6 6.2-.5 0-.9-1-1.2-2.7C13.8 21.6 13 24 12 24c-1 0-1.9-2.3-2.3-5.8-.3 1.7-.7 2.7-1.1 2.7-.8 0-1.4-2.6-1.6-6.2-.4 2-1 3.1-1.5 3.1-.8 0-1.5-2-1.8-4.9-.2.8-.5 1.3-.8 1.3-.6 0-1.2-2.8-1.2-6.3 0-3.4.6-6.2 1.2-6.2.4 0 .7.8 1 2C4.1 1.5 4.8 0 5.4 0c.7 0 1.4 2 1.7 5 .4-2.2.8-3.5 1.4-3.5.7 0 1.4 2.7 1.6 6.4.4-1.9 1-3.1 1.8-3.1.7 0 1.4 1.2 1.8 3.1.2-3.7.9-6.4 1.6-6.4.5 0 1 1.3 1.3 3.5.3-3 1-5 1.8-5 .7 0 1.2 1.4 1.6 3.6zM.7 10C.3 10 0 8.7 0 7.2c0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8S1.1 10 .7 10zm22.6 0c-.4 0-.7-1.3-.7-2.8 0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8s-.3 2.8-.7 2.8z"/></svg>'
  };
  function getSocials(data) {
    const socials = {};
    if (data.radio_facebook_url) {
      socials.facebook = data.radio_facebook_url;
    }
    if (data.radio_youtube) {
      socials.youtube = data.radio_youtube;
    }
    if (data.radio_instagram) {
      socials.instagram = data.radio_instagram;
    }
    if (data.radio_whatsapp) {
      socials.whatsapp = `https://wa.me/+${data.radio_whatsapp}`;
    }
    if (data.radio_menu_noticias) {
      socials.tiktok = data.radio_menu_noticias;
    }
    return socials;
  }

    function getApps(data) {
        const apps = {};
        // Verifica si existe el campo para la app Android
        if (data.radio_fondo_color) {
            apps.android = data.radio_fondo_color;  // Asigna la URL para la app Android
        }
        // Verifica si existe el campo para la app iOS
        if (data.radio_splash_color) {
            apps.ios = data.radio_splash_color;  // Asigna la URL para la app iOS
        }
        return apps;
    }




  // Convertir a tiempo relativo
  function timeAgo(date) {
    const getSecondsDiff = timestamp => (Date.now() - timestamp) / 1000;

    // Unidades de tiempo
    const DATE_UNITS = {
      day: 86400,
      hour: 3600,
      minute: 60,
      second: 1
    };

    // Obtener unidad y valor de la fecha
    const getUnitAndValueDate = secondsElapsed => {
      for (const [unit, secondsInUnit] of Object.entries(DATE_UNITS)) {
        if (secondsElapsed >= secondsInUnit || unit === 'second') {
          const value = Math.floor(secondsElapsed / secondsInUnit) * -1;
          return {
            value,
            unit
          };
        }
      }
    };

    // Obtener tiempo relativo
    const getTimeAgo = timestamp => {
      const rtf = new Intl.RelativeTimeFormat();
      const secondsElapsed = getSecondsDiff(timestamp);
      const {
        value,
        unit
      } = getUnitAndValueDate(secondsElapsed);
      return rtf.format(value, unit);
    };

    // Resultado
    const reference = new Date(date);
    return getTimeAgo(reference);
  }

  // Normaliza los datos de la estación
  // @param {object} data - Datos de la estación
  // @param {string} baseUrl - URL base de la API
  function getStationData(data, baseUrl, isMultiStream = false) {
    const station = {
      name: isMultiStream ? data.multiradio_nombre : data.radio_nombre,
        //description: isMultiStream ? data.multiradio_descripcion : data.radio_descripcion || 'Conectando',
      description: isMultiStream ? data.multiradio_descripcion : (data.radio_descripcion || data.radio_slogan),

      stream_url: isMultiStream ? data.multiradio_url : data.radio_url,
      tv_url: isMultiStream ? data.multiradio_tvstream : data.radio_video,
      picture: isMultiStream ? `${baseUrl}${data.multiradio_imagen}` : `${baseUrl}${data.radio_fondo}`,
      background: isMultiStream ? `${baseUrl}${data.multiradio_imagen}` : `${baseUrl}${data.radio_splash}`,
      api: isMultiStream ? data.multiradio_api : data.radio_menu_nicio
    };
    return station;
  }

  // Devolver una promesa para saber si la imagen se ha cargado correctamente
  // @param {string} src - URL de la imagen
  // @returns {Promise} - Promesa que se resuelve si la imagen se carga correctamente
  function loadImage(src) {
    return new Promise((resolve, reject) => {
      const img = new Image();
      img.onload = () => resolve(img);
      img.onerror = reject;
      img.src = src;
    });
  }

  // Agrega una transición de deslizamiento a la imagen
  // @param {HTMLElement} container - Contenedor de la imagen
  // @param {string} src - URL de la imagen
  function slideUpImageTransition(container, src) {
    const img = document.createElement('img');
    const size = container.clientHeight;
    img.src = src;
    img.width = size;
    img.height = size;
    container.appendChild(img);
    const firstImg = container.querySelector('img:first-child');
    firstImg.style.marginLeft = `-${size}px`;
    firstImg.addEventListener('transitionend', () => {
      const allImgExcLast = container.querySelectorAll('img:not(:last-child)');
      allImgExcLast.forEach(img => img.remove());
    }, {
      once: true
    });
  }

  // Agrega una transición de desvanecimiento a la imagen
  // @param {HTMLElement} container - Contenedor de la imagen
  // @param {string} src - URL de la imagen
  function fadeImageTransition(container, src) {
    container.style.opacity = 0;
    container.addEventListener('transitionend', () => {
      container.src = src;
      container.style.opacity = 0.2;
      container.addEventListener('transitionend', () => {
        container.removeAttribute('style');
      }, {
        once: true
      });
    });
  }

  // Crear una imagen proxy
  // @param {string} src - URL de la imagen
  // @returns {Promise} - Promesa que se resuelve con la imagen
  function createProxyImage(src) {
    return new Promise((resolve, reject) => {
      const img = document.createElement('img');
      img.crossOrigin = 'Anonymous';
      img.src = `https://images.weserv.nl/?url=${src}`;
      img.onload = () => resolve(img);
      img.onerror = reject;
    });
  }

  // Establecer el color de acento de la página
  // @param {HTMLImageElement} image - Imagen a extraer el color
  // @param {ColorThief} colorThief - Instancia de ColorThief
  function setAccentColor(image, colorThief) {
    const metaThemeColor = document.querySelector('meta[name=theme-color]');

    // Función para establecer el color
    const applyColor = () => {
      const color = `rgb(${colorThief.getColor(image)})`;
      document.querySelector('html').style.setProperty('--player-accent', color);
      if (metaThemeColor) {
        metaThemeColor.setAttribute('content', color);
      }
    };
    if (image.complete) {
      applyColor();
    } else {
      image.addEventListener('load', applyColor);
    }
  }

  // Set attributes
  // @param {HTMLElement} element - element
  // @param {Object} attributes - attributes
  // @returns {void}
  const setAttributes = (element, attributes) => {
    for (const key in attributes) {
      element.setAttribute(key, attributes[key]);
    }
  };

  // Create embed
  // @param {string} url - url
  // @param {Object} params - attributes
  // @returns {HTMLIFrameElement} - iframe element
  function createEmbed({
    url,
    params = {}
  }) {
    const $iframe = document.createElement('iframe');
    $iframe.src = url;
    $iframe.frameborder = 0;
    setAttributes($iframe, params);
    return $iframe;
  }

  // Get video info
  // @param {string} url - url
  // @returns {Promise<string>} - title
  async function getVideoInfo(videoId) {
    const data = await fetch(`https://noembed.com/embed?dataType=json&url=https://www.youtube.com/watch?v=${videoId}`);
    return data.json();
  }

  const modal = document.querySelector('.modal');
  const modalBody = modal.querySelector('.modal-body');
  const OPEN_CLASS = 'is-open';
  let currentVideo = null;
  function initVideoPlayer(videoUrl) {
    const $video = document.createElement('video');
    $video.id = 'player';
    $video.classList.add('video-js', 'video-js-stream');
    modalBody.innerHTML = '';
    modalBody.appendChild($video);
    const videoId = document.getElementById('player');

    // eslint-disable-next-line no-undef
    currentVideo = videojs(videoId, {
      sources: [{
        src: videoUrl,
        type: 'application/x-mpegURL'
      }],
      controls: true,
      autoplay: true
    });
  }
  function createShareButton(name, url) {
    const $a = document.createElement('a');
    $a.className = `player-social-link player-social-link-${name}`;
    $a.href = url;
    $a.target = '_blank';
    $a.innerHTML = svgIcons[name];
    return $a;
  }
  function initShareModal({
    artist,
    title,
    artwork
  }) {
    const shareUrl = window.location.href;
    const shareText = `Estoy escuchando ${title} de ${artist} en ${shareUrl}`;
    modalBody.innerHTML = `
    <h2 class="fw-800 mb-0.75">COMPARTIR WEBPLAYER</h2>
    <img src="${artwork}" alt="portada modal" class="player-modal-image" width="200" height="200">
    <div class="fs-4 fw-500 song-name">${title}</div>
    <span class="fs-5">${artist}</span>
    <div class="flex g-0.5 justify-center modal-social player-socials wrap">
      ${createShareButton('facebook', `https://www.facebook.com/sharer/sharer.php?u=${shareUrl}`).outerHTML}
      ${createShareButton('x', `https://twitter.com/intent/tweet?text=${encodeURIComponent(shareText)}`).outerHTML}
      ${createShareButton('whatsapp', `https://api.whatsapp.com/send?text=${encodeURIComponent(shareText)}`).outerHTML}
    </div>
  `;
  }
  function createForm(formAction, formId) {
    modalBody.innerHTML = `
    <form id="${formId}" action="${formAction}" method="POST" class="modal-form flex column g-1">
      <input type="text" name="name" placeholder="Nombre" required>
      <input type="email" name="email" placeholder="Correo" required>
      <input type="tel" name="contact" placeholder="Teléfono" required>
      <textarea name="message" placeholder="Mensaje" required></textarea>
      <input type="hidden" name="promo" value="${formId}">
      <button type="submit" class="btn btn-full btn-input">Enviar</button>
    </form>
  `;
    const $form = modalBody.querySelector('form');
    $form.addEventListener('submit', async e => {
      e.preventDefault();
      const formData = new FormData($form);
      const response = await fetch(formAction, {
        method: 'POST',
        body: formData
      });
      document.body.classList.add(response.ok ? 'modal-success' : 'modal-error');
      setTimeout(() => document.body.classList.remove('modal-success', 'modal-error'), 3000);
      if (response.ok) $form.reset();
    });
  }
  function openLyricsModal(lyrics) {
    modalBody.innerHTML = `
    <div class="lyrics-header">
      <h2 class="modal-lyrics-title m:fs-3 fs-5 fw-500">Letra</h2>
    </div>
    <pre>${lyrics || 'No se encontraron letras'}</pre>
  `;
  }
  function openSongModal({
    artist,
    title,
    artwork,
    genre,
    album
  }) {
    modalBody.innerHTML = `<div class="modal-song-info p-1 text-center">
    <img src="${artwork}" alt="portada modal" class="player-modal-image" width="200" height="200">
    <h3 class="fs-4 fw-700 color-text-inverse song-name">${title}</h3>
    <div class="flex column color-text-inverse justify-center modal-social player-socials wrap">
      <span class="fs-5"><strong class="color-text-inverse fw-700">Artista: </strong>${artist}</span>
      <span class="fs-5"><strong class="color-text-inverse fw-700">Genero: </strong>${genre}</span>
      <span class="fs-5"><strong class="color-text-inverse fw-700">Album: </strong>${album}</span>
    </div>
  </div>
  `;
  }
  function openFullModal({
    titulo,
    contenido,
    imagen,
    time
  }) {
    document.body.classList.add('modal-article');
    modalBody.innerHTML = `
    <div class="modal-article-title">
      <h2 class="fs-5 fw-800 m:fs-2">${titulo}</h2>
      <span class="fs-5">${time}</span>
    </div>
    <div class="modal-article-content">
      ${contenido}
    </div>
  `;
  }
  function openModal(type, options = {}) {
    if (!modal) return;
    modal.classList.add(OPEN_CLASS);
    modalBody.innerHTML = '';
    modal.classList.add(`modal-${type}`);
    switch (type) {
      case 'video':
        initVideoPlayer(options.url);
        break;
      case 'share':
        initShareModal(options.data);
        break;
      case 'form':
        createForm(options.formAction, options.formId);
        break;
      case 'lyrics':
        openLyricsModal(options.lyrics);
        break;
      case 'song':
        openSongModal(options.data);
        break;
      case 'article':
        openFullModal(options.data);
        break;
      default:
        console.warn('Tipo de modal desconocido:', type);
        return;
    }
    function closeModal() {
      modal.classList.remove(OPEN_CLASS, `modal-${type}`);
      document.body.classList.remove('modal-article');
      if (currentVideo) currentVideo.dispose();
      modalBody.innerHTML = '';
      if (options.hook && typeof options.hook === 'function') {
        options.hook();
      }
    }
    modal.querySelector('.modal-close').addEventListener('click', closeModal);
    modal.addEventListener('click', e => {
      if (!modal.querySelector('.modal-content').contains(e.target)) closeModal();
    });
  }

    let currentIframeActive = null;
    const cache$1 = localStorage.getItem('ytListMini') ? JSON.parse(localStorage.getItem('ytListMini')) : {};

    async function fetchVideoList(data, CONTAINER_LIST) {
        // Limpiar el contenedor para que no se acumulen videos
        CONTAINER_LIST.innerHTML = '';

        // Limitar a máximo 5 videos
        const items = data.slice(0, 5); // No es necesario el if, slice ya lo limita
        for (const item of items) {
            const videoInfo = cache$1[item.video_url] || (await getVideoInfo(item.video_url));
            const $li = createVideoListItem(item.video_url, videoInfo);
            CONTAINER_LIST.appendChild($li);
            addEventListeners($li, item.video_url);
            cache$1[item.video_url] = videoInfo;
            localStorage.setItem('ytListMini', JSON.stringify(cache$1));
        }

        // Reproducir el primer video automáticamente
        const $firstVideo = CONTAINER_LIST.querySelector('.ytlistmini-load-iframe');
        if ($firstVideo) {
            $firstVideo.click();
        }
    }

  
  function createVideoListItem(videoId, videoInfo) {
    const $div = document.createElement('div');
    const videoThumbnail = videoInfo.thumbnail_url.replace('hqdefault', 'mqdefault');
    $div.className = 'ytlistmini-item';
    $div.innerHTML = `
    <div class="ytlistmini-content" data-video-id="${videoId}">
      <div class="ytlistmini-element">
        <img src="${videoThumbnail}" alt="${videoInfo.title}">
        <div class="ytlistmini-info">
          <h3>${videoInfo.title}</h3>
          <p>${videoInfo.author_name}</p>
        </div>
        <button class="ytlistmini-load-iframe"></button>
      </div>
      <div class="ytlistmini-iframe"></div>
    </div>
  `;
    return $div;
  }
  function addEventListeners($li, videoId) {
    $li.querySelector('.ytlistmini-load-iframe').addEventListener('click', function () {
      handleLoadIframeButtonClick($li, videoId);
    });
  }
  function handleLoadIframeButtonClick($li, videoId) {
    const $iframe = createEmbed({
      url: `https://www.youtube.com/embed/${videoId}`,
      params: {
        allowfullscreen: '1',
        class: 'ytube-embed'
      }
    });
    if ($li.classList.contains('is-active')) return;
    $li.classList.add('is-active');
    if (currentIframeActive) {
      currentIframeActive.innerHTML = '';
      currentIframeActive.classList.remove('is-active');
      currentIframeActive.parentElement.parentElement.classList.remove('is-active');
    }
    const $iframeContainer = $li.querySelector('.ytlistmini-iframe');
    $iframeContainer.innerHTML = '';
    $iframeContainer.appendChild($iframe);
    currentIframeActive = $iframeContainer;
  }

  const TIMEOUT$1 = window?.streams?.sliderTiming || 5000;

  // @param {Object} data - Datos de la estación
  function setupStream(container, data) {
    if (!container || !data) return;
    const pictureEl = container.querySelector('.player-picture img:first-child');
    const backgroundEl = container.querySelector('.player-cover');
    const stationName = container.querySelector('.station-name');
    const stationDescription = container.querySelector('.station-description');
    const stationTv = container.querySelector('.station-tv');
    if (pictureEl) {
      pictureEl.src = data.picture;
    }
    if (backgroundEl) {
      backgroundEl.src = data.background;
    }
    if (stationName) {
      stationName.textContent = data.name;
    }
    if (stationDescription) {
      stationDescription.textContent = data.description;
    }
    if (stationTv) {
      stationTv.classList.remove('none');
    } else {
      stationTv.classList.add('none');
    }
  }

  // Establecer el slider
  // @param {HTMLElement} container - Contenedor del slider
  // @param {Array} data - Datos de las imágenes
  // @param {string} baseUrl - URL base de las imágenes
  function setupSlider(container, data, baseUrl) {
    const banners = data.map(item => {
      return `<div class="slider-slide">
      <a href="${item.slide_url}" target="_blank">
        <img src="${baseUrl}${item.slide_foto}" alt="Slide" class="slider-image" width="100%">
      </a>
    </div>`;
    });
    container.innerHTML = banners.join('');
    container.classList.add('is-loaded');
    if (container) {
      const SLIDES = data.length;

      // Mover el contenedor Wrapper dependiendo de la cantidad de slides
      let currentSlide = 0;
      let intervalId;
      function moveSlide() {
        if (currentSlide < SLIDES - 1) {
          container.style.transform = `translateX(-${(currentSlide + 1) * 100}%)`;
          currentSlide++;
        } else {
          container.style.transform = 'translateX(0)';
          currentSlide = 0;
        }
        intervalId = setTimeout(moveSlide, TIMEOUT$1);
      }
      function startSlider() {
        intervalId = setTimeout(moveSlide, TIMEOUT$1);
      }
      function stopSlider() {
        clearTimeout(intervalId);
      }
      container.addEventListener('mouseover', stopSlider);
      container.addEventListener('mouseout', startSlider);
      container.addEventListener('mousedown', stopSlider);
      container.addEventListener('mouseup', startSlider);
      setTimeout(moveSlide, TIMEOUT$1);
    }
  }

  // Establecer las redes sociales
  // @param {HTMLElement} container - Contenedor de las redes sociales
  // @param {object} data - Datos de las redes sociales
  function setupSocial(container, data) {
    if (!container || !data) return;
    const socialItems = Object.keys(data).map(key => {
      return `<a href="${data[key]}" target="_blank" class="player-social-link player-social-link-${key}">
      ${svgIcons[key]}
    </a>`;
    });
    container.innerHTML = socialItems.join('');
  }

  // Obtener los datos de la búsqueda
  // @param {string} query - Consulta de búsqueda
  // @param {string} service - Servicio de búsqueda
  // @returns {Promise<object>} - Datos de la búsqueda
  const cache = {};
  const getDataFromSearchApi = async (query, service) => {
    if (cache[query]) {
      return cache[query];
    }
      const streamUrl = `https://nexoapi.lat/search?query=${encodeURIComponent(query)}&service=${service}`;
    const response = await fetch(streamUrl);
    const data = await response.json();

    // Si no responde
    if (!data.results) {
      return {};
    }
    const results = data.results;
    cache[query] = results;
    return results;
  };

  // Obtener datos del stream
  // @param {string} streamUrl - URL del stream
  // @returns {Promise<object>} - Datos del stream
  async function getStreamData(streamUrl) {
    if (!streamUrl) {
      return {};
    }

    // const jsonUri = `https://free.radioapi.lat/nx?url=${encodeURIComponent(streamUrl)}`
      const jsonUri = `https://free.radioapi.lat/met?url=${encodeURIComponent(streamUrl)}`;
    // const jsonUri = `https://free.radioapi.lat/nexo?url=${encodeURIComponent(streamUrl)}`

    try {
      const response = await fetch(jsonUri);
      return response.json();
    } catch (error) {
      console.error('Error al obtener los datos del stream:', error);
      return {};
    }
  }

  // Obtener letras de canciones usando Lyrics.ovh
  const getLyricsOvh = async (artist, name) => {
    try {
      const response = await fetch(`https://api.lyrics.ovh/v1/${encodeURIComponent(artist)}/${encodeURIComponent(name)}?auto=1`);
      if (!response.ok) throw new Error('No se pudo obtener la letra');
      const data = await response.json();
      if (data.lyrics) {
        return data.lyrics;
      } else {
        return 'Letra no disponible';
      }
    } catch (error) {
      console.warn('No se pudo obtener la letra de Lyrics.ovh');
      return 'Letra no disponible';
    }
  };

  // obtener letras de vagalume
  const getVagalumeLyrics = async (artist, name) => {
    const apikey = '1637b78dc3b129e6843ed674489a92d0';
    try {
      const response = await fetch(`https://api.vagalume.com.br/search.php?apikey=${apikey}&art=${encodeURIComponent(artist)}&mus=${encodeURIComponent(name)}`);
      if (!response.ok) throw new Error('No se pudo obtener la letra');
      const data = await response.json();
      if (data.type === 'notfound') {
        return 'Letra no disponible';
      }
      return data.mus[0].text;
    } catch (error) {
      console.warn('No se pudo obtener la letra de Vagalume');
      return 'Letra no disponible';
    }
  };

  // obtener letras
  const getLyrics = async (artist, name) => {
    let lyrics = await getLyricsOvh(artist, name);
    if (!lyrics || lyrics === 'Letra no disponible') {
      lyrics = await getVagalumeLyrics(artist, name);
    }
    return lyrics;
  };

  const player = document.querySelector('.player');
  const playerSocial = document.querySelector('.player-social');
  const playButton = document.querySelector('.player-button-play');
  const visualizerContainer = document.querySelector('.visualizer');
  const isWave = localStorage.getItem('isWave') === 'true' || false;
  const rhythm = document.querySelector('.player-button-rhythm-toggle');

  // --- NUEVA VARIABLE DE TOKEN ---
  // Este token se incrementa cada vez que se inicia una nueva actualización de estación.
  let currentUpdateToken = 0;
  const STREAMS = window?.streams || {};
  const TIMEOUT = STREAMS?.timeRefresh || 5000;
  const SERVICE = STREAMS?.service || 'spotify';
  const MODULE_VIDEO_TOPS = STREAMS?.module_video_tops;
  const MODULE_NEWS = STREAMS?.module_news;
  const MODULE_TEAM = STREAMS?.module_team;
  const CAROUSEL_TIME = STREAMS?.carouselTiming || 5000;
  if (isWave) {
    visualizerContainer.classList.add('is-wave');
    rhythm.classList.add('is-wave');
  }
  rhythm.addEventListener('click', () => {
    rhythm.classList.toggle('is-wave');
    visualizerContainer.classList.toggle('is-wave');
    localStorage.setItem('isWave', visualizerContainer.classList.contains('is-wave'));
  });
  const audio = new Audio();
  let hasVisualizer = false;
  audio.crossOrigin = 'anonymous';
  if (playButton !== null) {
    playButton.addEventListener('click', async () => {
      if (audio.paused) {
        play(playButton);
      } else {
        pause(playButton);
      }
    });
  }
  function play(button, newSource = null) {
    if (newSource) {
      audio.src = newSource;
    }
    audio.load();
    audio.play();
    button.innerHTML = `${svgIcons.pause} Pausar`;
    button.classList.add('is-active');
    document.body.classList.add('is-playing');

    // Visualizer
    if (!hasVisualizer) {
      visualizer(audio, visualizerContainer);
      hasVisualizer = true;
    }
  }
  function pause(button) {
    audio.pause();
    button.innerHTML = `${svgIcons.play} Reproducir`;
    button.classList.remove('is-active');
    document.body.classList.remove('is-playing');
  }

  // Botón de volumen
  const range = document.querySelector('.player-volume');
  const rangeFill = document.querySelector('.player-range-fill');
  const rangeWrapper = document.querySelector('.player-range-wrapper');
  const rangeThumb = document.querySelector('.player-range-thumb');
  const volumeToggle = document.querySelector('.player-button-volume-toggle');
  const currentVolume = localStorage.getItem('volume') || 100;

  // Rango recorrido
  function setRangeWidth(percent) {
    {
      rangeFill.style.height = `${percent}%`;
    }
  }

  // Posición del thumb
  function setThumbPosition(percent) {
    const compensatedWidth = rangeWrapper.offsetHeight - rangeThumb.offsetWidth ;
    const thumbPosition = percent / 100 * compensatedWidth;
    {
      rangeThumb.style.bottom = `${thumbPosition}px`;
    }
  }

  // Actualiza el volumen al cambiar el rango
  function updateVolume(value) {
    range.value = value;
    setRangeWidth(value);
    setThumbPosition(value);
    localStorage.setItem('volume', value);
    audio.volume = value / 100;
    if (value === 0) {
      volumeToggle.innerHTML = icons.volumeMute;
    } else {
      volumeToggle.innerHTML = icons.volumeUp;
    }
  }

  // Valor inicial
  if (range !== null) {
    updateVolume(currentVolume);

    // Escucha el cambio del rango
    range.addEventListener('input', event => {
      updateVolume(event.target.value);
    });

    // Escucha el movimiento del mouse
    rangeThumb.addEventListener('mousedown', () => {
      document.addEventListener('mousemove', handleThumbDrag);
    });
  }

  // Intercambiar entre 0% y 100% de volumen
  if (volumeToggle !== null) {
    volumeToggle.addEventListener('click', () => {
      if (audio.volume > 0) {
        updateVolume(0);
      } else {
        updateVolume(100);
      }
    });
  }

  // Cargar datos de la canción actual al navegador
  function setMediaSession(data, picture, token) {
    // Si en algún momento el token no coincide, se descarta esta actualización.
    if (token !== currentUpdateToken) return;
    const {
      artist,
      artwork
    } = data;
    const title = data.title || data.song;
    if ('mediaSession' in navigator) {
      navigator.mediaSession.metadata = new MediaMetadata({
        title,
        artist,
        artwork: [{
          src: artwork || picture,
          sizes: '128x128',
          type: 'image/png'
        }, {
          src: artwork || picture,
          sizes: '256x256',
          type: 'image/png'
        }, {
          src: artwork || picture,
          sizes: '512x512',
          type: 'image/png'
        }]
      });
      navigator.mediaSession.setActionHandler('play', () => {
        play();
      });
      navigator.mediaSession.setActionHandler('pause', () => {
        pause();
      });
    }
  }

  // Mueve el thumb y actualiza el volumen
  function handleThumbDrag(event) {
    const rangeRect = range.getBoundingClientRect();
    const clickX = event.clientY - rangeRect.top;
    let percent = clickX / range.offsetWidth * 100;
    percent = 100 - percent; // Invertir el porcentaje si es vertical

    percent = Math.max(0, Math.min(100, percent));
    const value = Math.round((range.max - range.min) * (percent / 100)) + parseInt(range.min);
    updateVolume(value);
  }

  // Deja de escuchar el movimiento del mouse
  document.addEventListener('mouseup', () => {
    document.removeEventListener('mousemove', handleThumbDrag);
  });
  window.addEventListener('resize', () => {
    const currentPercent = range.value;
    setRangeWidth(currentPercent);
    setThumbPosition(currentPercent);
  });

  // Establecer datos de la canción actual
  function setCurrentSong(data, picture, token) {
    // Si en algún momento el token no coincide, se descarta esta actualización.
    if (token !== currentUpdateToken) return;
    const {
      artist,
      artwork
    } = data;
    const title = data.title || data.song;
    const songName = document.querySelector('.song-name');
    const songArtist = document.querySelector('.song-artist');
    const pageTitle = document.querySelector('title');
    if (songName) {
      songName.textContent = title;
    }
    if (songArtist) {
      songArtist.textContent = artist;
    }
    if (pageTitle) {
      pageTitle.textContent = `${title} - ${artist}`;
    }
    const pictureEl = document.querySelector('.player-picture');
    const artworkUrl = artwork || picture;

    // eslint-disable-next-line no-undef
    const colorThief = new ColorThief();

    // Establecer el color de acento
    createProxyImage(artworkUrl).then(img => {
      setAccentColor(img, colorThief);
    }).catch(() => {
      console.error('Error al cargar la imagen');
    });
    if (pictureEl) {
      loadImage(artworkUrl).then(() => {
        // Verifica nuevamente el token antes de aplicar la transición
        if (token === currentUpdateToken) {
          slideUpImageTransition(pictureEl, artworkUrl);
        }
      }).catch(() => {
        console.error('Error al cargar arte de la canción');
      });
    }
    const coverEl = document.querySelector('.player-cover');
    if (coverEl) {
      const coverUrl = artwork || picture;
      const $img = document.createElement('img');
      $img.src = coverUrl;
      loadImage(coverUrl).then(() => {
        if (token === currentUpdateToken) {
          fadeImageTransition(coverEl, coverUrl);
        }
      }).catch(() => {
        console.error('Error al cargar la portada de la canción');
      });
    }
  }

  // Establecer las canciones que se han reproducido
  async function setHistory(data, current, token) {
    // Si en algún momento el token no coincide, se descarta esta actualización.
    if (token !== currentUpdateToken) return;
    const historyContainer = document.querySelector('.last-played');
    if (!historyContainer || !data) return;
    const historyItems = historyContainer.querySelectorAll('.history').length;
    data = data.slice(0, historyItems);

    // Array para almacenar los datos de las canciones
    const historyCurrentData = [];
    const itemsHTML = await Promise.all(data.map(async ({
      song: {
        artist,
        title
      }
    }, index) => {
      const dataFrom = await getDataFromSearchApi(`${artist} - ${title}`, SERVICE);
        if (dataFrom.artwork.includes('https://nexoapi.lat/') || dataFrom.artwork === null || title.trim() === '') {
        dataFrom.artwork = current.picture || current.cover;
        dataFrom.title = current.name;
        dataFrom.artist = current.description;
      } else {
        dataFrom.title = title;
        dataFrom.artist = artist;
      }

      // Almacenar los datos en el índice correcto
      historyCurrentData[index] = dataFrom;
      return `
        <div class="history" data-index="${index}">
          <div class="history-wrapper">
            <img class="history-art" src="${dataFrom.artwork}" alt="${dataFrom.title}">
            <div class="history-meta fs-7">
              <span class="history-title fw-700 uppercase">${dataFrom.title}</span>
              <span class="history-artist">${dataFrom.artist}</span>
            </div>
          </div>
          <a class="history-url history-deezer" href="${dataFrom.stream || '#not-found'}" target="_blank">
            ${svgIcons.deezer}
          </a>
          <button class="history-button"></button>
        </div>`;
    }));
    if (token === currentUpdateToken) {
      historyContainer.innerHTML = itemsHTML.join('');
      const historyItemsElements = historyContainer.querySelectorAll('.history');
      historyItemsElements.forEach(item => {
        const button = item.querySelector('.history-button');
        const index = item.getAttribute('data-index');
        button.addEventListener('click', event => {
          // Abrir modal con la canción seleccionada
          event.preventDefault();
          openModal('song', {
            data: historyCurrentData[index]
          });
        });
      });
    }
  }
  function setArticles(data, baseUrl) {
    const newsContainer = document.getElementById('news');
    if (!newsContainer || !data) return;
    data = data.slice(0, 8);
    for (const item of data) {
      const {
        titulo,
        photo,
        contenido
      } = item;
      const time = new Date(item.created_at);
      const createdAt = timeAgo(time);
      const article = document.createElement('article');
      article.classList.add('news-item', 'swiper-slide');
      article.innerHTML = `
      <div class="news-image">
        <img src="${baseUrl}${photo}" alt="${titulo}">
      </div>
      <div class="news-content">
        <h3 class="news-title">${titulo}</h3>
        <time class="news-date">${createdAt}</time>
      </div>
    `;
      article.addEventListener('click', () => {
        openModal('article', {
          data: {
            titulo,
            contenido,
            imagen: `${baseUrl}${photo}`,
            time: createdAt
          }
        });
      });
      newsContainer.appendChild(article);
    }

    // eslint-disable-next-line no-undef, no-unused-vars
    new Swiper('.swiper-news', {
      // Enabled autoplay mode
      autoplay: {
        delay: CAROUSEL_TIME,
        disableOnInteraction: false
      },
      // If we need navigation
      navigation: {
        nextEl: '.swiper-next-news',
        prevEl: '.swiper-prev-news'
      },
      // Responsive breakpoints
      breakpoints: {
        480: {
          slidesPerView: 2,
          spaceBetween: 12
        },
        640: {
          slidesPerView: 3,
          spaceBetween: 12
        },
        1024: {
          slidesPerView: 4,
          spaceBetween: 12
        }
      }
    });
  }
  function setTeam(data, baseUrl) {
    const teamContainer = document.getElementById('team');
    if (!teamContainer || !data) return;
    data = data.slice(0, 8);
    for (const item of data) {
      const equipoNombre = item.equipo_nombre;
      const equipoFoto = item.equipo_foto;
      const equipoDescripcion = item.equipo_descripcion;
      const article = document.createElement('article');
      article.classList.add('team-item', 'swiper-slide');
      article.innerHTML = `
      <div class="team-image">
        <img src="${baseUrl}${equipoFoto}" alt="${equipoNombre}">
      </div>
      <div class="team-content">
        <h3 class="team-title">${equipoNombre}</h3>
        <p class="team-description">${equipoDescripcion}</p>
      </div>
    `;
      teamContainer.appendChild(article);
    }

    // eslint-disable-next-line no-undef, no-unused-vars
    new Swiper('.swiper-team', {
      // Enabled autoplay mode
      autoplay: {
        delay: CAROUSEL_TIME,
        disableOnInteraction: false
      },
      // If we need navigation
      navigation: {
        nextEl: '.swiper-next-team',
        prevEl: '.swiper-prev-team'
      },
      // Responsive breakpoints
      breakpoints: {
        480: {
          slidesPerView: 2,
          spaceBetween: 12
        },
        640: {
          slidesPerView: 3,
          spaceBetween: 12
        },
        1024: {
          slidesPerView: 4,
          spaceBetween: 12
        }
      }
    });
  }

  // ----------------
  // Iniciar Datos
  let timeoutId;
  let timeoutIdProgram;
  let currentSongPlaying = 'none';
  let loadStations = [];
  let currentStation;
  let currentActiveStation;
  const json = window.streams || {};
  const baseUrl = json.base_url || '';
  const urlServer = json.url_server || 'api/';
  const idUser = json.id_user || '';
  const isMultiStream = json.multi_stream || false;
  const stationsContainer = document.getElementById('player-prm');
  const stationsButton = document.querySelector('.station-prm');
  const radioPath = isMultiStream ? 'multiradio-app/' : 'radio-app/';
  const dataUrl = `${baseUrl}${urlServer}${radioPath}${idUser}`;
  const dataSocialUrl = `${baseUrl}${urlServer}radio-app/${idUser}`;
  const dataAppsUrl = `${baseUrl}${urlServer}radio-app/${idUser}`;
  const dataProgramUrl = `${baseUrl}${urlServer}programa-app/${idUser}`;
  const dataBannersUrl = `${baseUrl}${urlServer}slide-app/${idUser}`;
  const dataPromoUrl = `${baseUrl}${urlServer}promocion-app/${idUser}`;
  const dataGetPromoUrl = `${baseUrl}${urlServer}promotional-contact`;
  const dataNoticiasUrl = `${baseUrl}${urlServer}noticia-app/${idUser}`;
  const dataVideoUrl = `${baseUrl}${urlServer}video-app/${idUser}`;
  const dataTeamUrl = `${baseUrl}${urlServer}equipo-app/${idUser}`;
  const stationTv = document.querySelector('.station-tv');
  function setupAll(station) {
    audio.src = station.stream_url;
    setupStream(player, station);
    if (station.tv_url) {
      stationTv.onclick = () => {
        openModal('video', {
          url: station.tv_url,
          hook: () => play(playButton)
        });
        pause(playButton);
      };
    }
  }
  function setDeviceInfo() {
    const icons = {
      windows: '<i class="ri-microsoft-fill"></i>',
      android: '<i class="ri-android-fill"></i>',
      apple: '<i class="ri-apple-fill"></i>'
    };
    if (navigator.userAgent.match(/Android/i)) {
      return `${icons.android} Instalar en Android`;
    }
    if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {
      return `${icons.apple} Instalar en iOS`;
    }
    return `${icons.windows} Instalar en Windows`;
  }

  // Función para instalar PWA en el navegador
  function installPWA() {
    const container = document.getElementById('install-container');
    const close = document.getElementById('close-install');
    const install = document.getElementById('install-pwa');
    const checkIosDevice = () => {
      const ua = navigator.userAgent;
      const vendor = navigator.vendor;
      const isSafari = /Safari/.test(ua) && /Apple Computer/.test(vendor) && !/CriOS/.test(ua); // Excluir Chrome

      return isSafari;
    };
    if (checkIosDevice()) return;
    if (!container) return;
    if (!install) return;
    close.addEventListener('click', () => {
      container.classList.remove('is-active');
    });
    install.innerHTML = setDeviceInfo();
    const isMobileDevice = () => {
      const mobileDeviceRegex = /Android|Windows Phone|Windows|iPhone|iPad|iPod|iOS|Macintosh/i;
      return mobileDeviceRegex.test(navigator.userAgent);
    };
    const isFirefox = () => {
      return navigator.userAgent.includes('Firefox');
    };
    let installPrompt = null;
    window.addEventListener('beforeinstallprompt', event => {
      event.preventDefault();
      installPrompt = event;
      if (isMobileDevice() && !isFirefox()) {
        container.classList.add('is-active');
      }
    });
    install.addEventListener('click', async () => {
      if (!installPrompt) {
        console.warn('No se puede instalar la PWA');
        return;
      }
      const result = await installPrompt.prompt();
      if (result.outcome === 'accepted') {
        console.warn('PWA instalada');
        container.classList.remove('is-active');
      } else {
        console.warn('PWA no instalada');
      }
    });
  }
  installPWA();
  function convertSecondsToMinutes(seconds) {
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = seconds % 60;
    return `${minutes}:${remainingSeconds.toFixed(0) < 10 ? '0' : ''}${remainingSeconds.toFixed(0)}`;
  }

  // Obtener porcentaje entre dos valores
  function getPercentage(value, total) {
    return value * 100 / total;
  }
  function songProgress(duration, elapsed) {
    const progress = getPercentage(elapsed, duration);

    // Agregar la variable css con el progreso de la canción
    player.style.setProperty('--song-progress', `${2 + progress}%`);

    // Actualizar el progreso de la canción
    const songElapsed = document.querySelector('.song-elapsed');
    const songDuration = document.querySelector('.song-duration');
    if (songElapsed) {
      songElapsed.textContent = convertSecondsToMinutes(elapsed);
    }
    if (songDuration) {
      songDuration.textContent = convertSecondsToMinutes(duration);
    }
  }

  // Obtener datos de la estación
  async function initApp() {
    const response = await fetch(dataUrl);
    const data = await response.json();
    if (isMultiStream) {
      const stations = data.map(station => getStationData(station, baseUrl, true));
      loadStations = stations;
      setupAll(loadStations[0]);
    } else {
      const station = getStationData(data, baseUrl);
      currentStation = station;
      setupAll(station);
    }

    // Iniciar el stream
    async function init(current) {
      // Incrementar el token de actualización
      const token = ++currentUpdateToken;
      currentStation = current;

      // Cancelar el timeout anterior
      if (timeoutId) clearTimeout(timeoutId);
      const song = await getStreamData(current.stream_url);

      // Si en el intento se cambió la estación, se descarta esta respuesta
      if (token !== currentUpdateToken) return;
      if (song.now_playing) {
        songProgress(song.now_playing.duration, song.now_playing.elapsed);
      }
      const title = song.title || song.song;
      const query = title.trim() !== '' ? `${song.artist} - ${title}` : `${current.name} - ${current.descripcion}`;
      if (currentSongPlaying !== query) {
        currentSongPlaying = query;
        const songData = await getDataFromSearchApi(query, SERVICE);
          if (songData.artwork.includes('https://nexoapi.lat/') || songData.artwork === null || title.trim() === '') {
          songData.artwork = current.picture || current.cover;
          songData.title = current.name;
          songData.artist = current.description;
        } else {
          songData.title = title;
          songData.artist = song.artist;
        }
        setCurrentSong(songData, currentStation.picture, token);
        setMediaSession(songData, currentStation.picture, token);
        setHistory(song.song_history, currentStation, token);
        const buttonShare = document.querySelector('.player-button-share');
        if (buttonShare) {
          buttonShare.onclick = () => {
            openModal('share', {
              data: song
            });
          };
        }
        const lyricsData = await getLyrics(song.artist, title);
        const lyricsButton = document.querySelector('.player-button-lyrics');
        if (lyricsButton) {
          lyricsButton.onclick = () => {
            openModal('lyrics', {
              lyrics: lyricsData
            });
          };
        }
      }
      timeoutId = setTimeout(() => {
        init(current);
      }, TIMEOUT);
    }

    // DATOS DE LA PROGRAMACIÓN
    function initProgram() {
      if (timeoutIdProgram) clearTimeout(timeoutIdProgram);
      fetch(dataProgramUrl).then(response => response.json()).then(data => {
        setupSchedule(data, baseUrl);
      }).catch(error => {
        console.error('Error:', error);
      });
      // Refrescar cada 1 minuto
      timeoutIdProgram = setTimeout(() => {
        initProgram();
      }, 60000);
    }
    initProgram();

    // DATOS DE LOS BANNERS
    fetch(dataBannersUrl).then(response => response.json()).then(data => {
      const container = document.querySelector('.slider-wrapper');
      setupSlider(container, data, baseUrl);
    }).catch(error => {
      console.error('Error:', error);
    });

    // Botones sociales
    fetch(dataSocialUrl).then(response => response.json()).then(data => {
      setupSocial(playerSocial, getSocials(data));
    }).catch(error => {
      console.error('Error:', error);
    });

      // Botones Apps
      fetch(dataAppsUrl)
          .then(response => response.json())
          .then(data => {
              setupApps(getApps(data)); // Llama a la función setupApps con el resultado de getApps
          })
          .catch(error => {
              console.error('Error:', error);
          });

      function setupApps(apps) {
          // Verifica si hay URL para Android y actualiza el botón
          if (apps.android) {
              const androidButton = document.getElementById('androidButton');
              androidButton.href = apps.android; // Asigna la URL para Android
              androidButton.style.display = 'inline-block'; // Asegura que se muestre el botón
          } else {
              const androidButton = document.getElementById('androidButton');
              androidButton.style.display = 'none'; // Oculta el botón si no hay URL
          }

          // Verifica si hay URL para iOS y actualiza el botón
          if (apps.ios) {
              const iosButton = document.getElementById('iosButton');
              iosButton.href = apps.ios; // Asigna la URL para iOS
              iosButton.style.display = 'inline-block'; // Asegura que se muestre el botón
          } else {
              const iosButton = document.getElementById('iosButton');
              iosButton.style.display = 'none'; // Oculta el botón si no hay URL
          }
      }



    const promoContainer = document.getElementById('player-prm-send');
    if (promoContainer) {
      // DATOS DE LAS PROMOCIONES
      fetch(dataPromoUrl).then(response => response.json()).then(data => {
        if (data.length === 0) return;
        const button = document.querySelector('.send-prm');
        button.classList.remove('none');
        data.forEach(item => {
          const img = document.createElement('img');
          img.src = baseUrl + item.photo;
          img.classList.add('prm-image');
          img.addEventListener('click', () => {
            openModal('form', {
              formAction: dataGetPromoUrl,
              formId: item.promocion_id
            });
          });
          promoContainer.appendChild(img);
        });
      }).catch(error => {
        console.error('Error:', error);
      });
    }

    // DATOS DE LAS NOTICIAS
    function initNoticias() {
      const newsContainer = document.querySelector('.main-news');
      if (!newsContainer) return;
      fetch(dataNoticiasUrl).then(response => response.json()).then(data => {
        if (data.length === 0) return;
        setArticles(data, baseUrl);
        newsContainer.classList.remove('none');
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    if (MODULE_NEWS) initNoticias();

    // DATOS DE LAS NOTICIAS
    function initTeam() {
      const teamContainer = document.querySelector('.main-team');
      if (!teamContainer) return;
      fetch(dataTeamUrl).then(response => response.json()).then(data => {
        if (data.length === 0) return;
        setTeam(data, baseUrl);
        teamContainer.classList.remove('none');
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    if (MODULE_TEAM) initTeam();
    function initVideoApp() {
      const container = document.querySelector('.main-ytlist');
      if (!container) return;
      const CONTAINER_LIST = container.querySelector('.ytlistmini-list');
      if (!CONTAINER_LIST) return;
      fetch(dataVideoUrl).then(response => response.json()).then(data => {
        container.classList.remove('none');
        fetchVideoList(data, CONTAINER_LIST);
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    if (MODULE_VIDEO_TOPS) initVideoApp();

    // Estaciones
    if (loadStations.length > 1) {
      stationsButton.classList.remove('none');
      loadStations.forEach((station, index) => {
        const img = document.createElement('img');
        img.classList.add('prm-image');
        img.src = station.picture;
        if (index === 0) {
          img.classList.add('is-current');
          currentActiveStation = img;
        }
        img.addEventListener('click', event => {
          if (currentStation === station) return;
          if (currentActiveStation) {
            currentActiveStation.classList.remove('is-current');
          }
          currentActiveStation = img;
          currentActiveStation.classList.add('is-current');
          init(station);
          setupAll(station);
          play(playButton, station.stream_url);
          currentStation = station;
        });
        stationsContainer.appendChild(img);
      });
    }

    // Iniciar todo
    if (isMultiStream) {
      init(loadStations[0]);
    } else {
      init(currentStation);
    }
  }
  initApp();

})();
