(function () {
  'use strict';

  function setDeviceInfo() {
    const icons = {
      windows: '<i class="ri-microsoft-fill"></i>',
      android: '<i class="ri-android-fill"></i>',
      apple: '<i class="ri-apple-fill"></i>'
    };
    if (navigator.userAgent.match(/Android/i)) {
      return `${icons.android} Instalar en Android`;
    }
    if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {
      return `${icons.apple} Instalar en iOS`;
    }
    return `${icons.windows} Instalar en Windows`;
  }
  function installPWA() {
    const container = document.getElementById('install-container');
    const close = document.getElementById('close-install');
    const install = document.getElementById('install-pwa');
    const checkIosDevice = () => {
      const ua = navigator.userAgent;
      const vendor = navigator.vendor;
      const isSafari = /Safari/.test(ua) && /Apple Computer/.test(vendor) && !/CriOS/.test(ua); // Excluir Chrome

      return isSafari;
    };
    if (checkIosDevice()) return;
    if (!container) return;
    if (!install) return;
    close.addEventListener('click', () => {
      container.classList.remove('is-active');
    });
    install.innerHTML = setDeviceInfo();
    const isMobileDevice = () => {
      const mobileDeviceRegex = /Android|Windows Phone|Windows|iPhone|iPad|iPod|iOS|Macintosh/i;
      return mobileDeviceRegex.test(navigator.userAgent);
    };
    const isFirefox = () => {
      return navigator.userAgent.includes('Firefox');
    };
    let installPrompt = null;
    window.addEventListener('beforeinstallprompt', event => {
      event.preventDefault();
      installPrompt = event;
      console.log('beforeinstallprompt', installPrompt);
      if (isMobileDevice() && !isFirefox()) {
        container.classList.add('is-active');
      }
    });
    install.addEventListener('click', async () => {
      if (!installPrompt) {
        console.warn('No se puede instalar la PWA');
        return;
      }
      const result = await installPrompt.prompt();
      if (result.outcome === 'accepted') {
        console.warn('PWA instalada');
        container.classList.remove('is-active');
      } else {
        console.warn('PWA no instalada');
      }
    });
  }

  const STREAMS = window?.streams || {};
  const urlPanel = `${STREAMS?.base_url}api/radio-app/${STREAMS?.id_user}`;

  // Expresiones regulares para validar URLs externas o base64
  const EXTERNAL_URL_REGEX = /^(?:https?:)?\/\/|^(?:data|blob):/i;
  const EXTERNAL_RESOURCE_REGEX = /^(https?:)?\/\/.*/i;

  // Función para ajustar las rutas de los elementos
  function adjustElementPaths(elements, attribute, basePath) {
    elements.forEach(el => {
      const value = el.getAttribute(attribute);
      if (value && !EXTERNAL_URL_REGEX.test(value)) {
        el.setAttribute(attribute, `${basePath}/${value}`);
      }
    });
  }

  // Función para procesar y agregar elementos al head
  function processHeadElements(headElements, basePath) {
    const loadPromises = [];
    headElements.forEach(element => {
      let newElement;
      let promise;
      if (element.tagName === 'LINK' && (element.rel === 'stylesheet' || element.rel === 'preload')) {
        const href = element.getAttribute('href');
        newElement = document.createElement('link');
        newElement.rel = element.rel;
        newElement.href = EXTERNAL_RESOURCE_REGEX.test(href) ? href : `${basePath}/${href}`;
        promise = new Promise(resolve => {
          newElement.onload = resolve;
          newElement.onerror = resolve; // Considerar error como cargado para evitar bloqueo
        });
      } else if (element.tagName === 'SCRIPT' && element.src) {
        const src = element.getAttribute('src');
        newElement = document.createElement('script');
        newElement.src = EXTERNAL_RESOURCE_REGEX.test(src) ? src : `${basePath}/${src}`;
        newElement.defer = true;
        promise = new Promise(resolve => {
          newElement.onload = resolve;
          newElement.onerror = resolve;
        });
      } else if (element.tagName === 'META' && element.name !== 'viewport') {
        newElement = document.createElement('meta');
        newElement.name = element.name;
        newElement.content = element.content;
      }
      if (newElement) {
        document.head.appendChild(newElement);
        if (promise) loadPromises.push(promise);
      }
    });
    return Promise.all(loadPromises);
  }

  // Función para cargar el reproductor en el contenedor
  async function loadPlayer(playerFile) {
    const basePath = `players/${playerFile}`;
    const response = await fetch(`${basePath}/index.html`);
    if (!response.ok) {
      throw new Error('No se pudo cargar el reproductor');
    }
    const html = await response.text();
    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');

    // Ajustar rutas dentro del body
    const bodyElements = doc.body.querySelectorAll('[src], [href]');
    adjustElementPaths(bodyElements, 'src', basePath);
    adjustElementPaths(bodyElements, 'href', basePath);

    // Insertar el contenido procesado en el contenedor principal
    document.getElementById('app').innerHTML = doc.body.innerHTML;

    // Procesar los enlaces CSS y scripts JS
    const headElements = doc.head.querySelectorAll('link, script');
    processHeadElements(headElements, basePath).then(() => {
      document.body.classList.remove('preloader');
    });
  }
  async function initPanel(players) {
    const response = await fetch(urlPanel);
    const data = await response.json();
    const playerName = STREAMS?.player_name || data?.radio_url_rss;
    await loadPlayer(playerName);
    const player = players[playerName] || players.mainplayer;
    if (player) {
      player();
    } else {
      console.error('Reproductor no encontrado');
    }
  }

  function mainplayer() {
    window.addEventListener('DOMContentLoaded', () => {
      const body = document.querySelector('body');
      body.classList.remove('preload');
    });
    const buttons = document.querySelectorAll('[data-outside]');
    const ACTIVE_CLASS = 'is-active';
    function outsideClick(button) {
      if (!button) return;
      const target = document.getElementById(button.dataset.outside);
      if (!target) return;
      function toggleClasses() {
        button.classList.toggle(ACTIVE_CLASS);
        target.classList.toggle(ACTIVE_CLASS);
        if (button.classList.contains(ACTIVE_CLASS)) {
          document.addEventListener('click', clickOutside);
          return;
        }
        document.removeEventListener('click', clickOutside);
      }
      button.addEventListener('click', toggleClasses);
      function clickOutside(event) {
        if (!target.contains(event.target) && !button.contains(event.target)) {
          toggleClasses();
          document.removeEventListener('click', clickOutside);
        }
      }
      const closeButton = target.querySelector('[data-close]');
      if (closeButton) {
        closeButton.addEventListener('click', () => {
          button.classList.remove(ACTIVE_CLASS);
          target.classList.remove(ACTIVE_CLASS);
          document.removeEventListener('click', clickOutside);
        });
      }
    }
    buttons.forEach(button => {
      outsideClick(button);
    });
    const tiltBox = document.querySelector('.tilt-box');
    const rect = tiltBox.getBoundingClientRect();
    const maxTilt = 12; // Cantidad maxima de inclinacion
    const extraMargin = 0; // Margen adicional en pixeles alrededor del cuadro

    const expandedRect = {
      left: rect.left - extraMargin,
      right: rect.right + extraMargin,
      top: rect.top - extraMargin,
      bottom: rect.bottom + extraMargin,
      width: rect.width + extraMargin * 2,
      height: rect.height + extraMargin * 2
    };

    // Listener dentro del área expandida
    document.addEventListener('mousemove', e => {
      const mouseX = e.clientX;
      const mouseY = e.clientY;

      // Verificamos si el cursor está dentro del área expandida
      if (mouseX >= expandedRect.left && mouseX <= expandedRect.right && mouseY >= expandedRect.top && mouseY <= expandedRect.bottom) {
        const x = mouseX - expandedRect.left;
        const y = mouseY - expandedRect.top;
        const halfWidth = expandedRect.width / 2;
        const halfHeight = expandedRect.height / 2;

        // Calculamos la inclinación basada en la posición del mouse en el área expandida
        const tiltX = (x - halfWidth) / halfWidth * maxTilt;
        const tiltY = -((y - halfHeight) / halfHeight) * maxTilt;

        // Aplicamos la transformación
        tiltBox.style.transform = `perspective(1000px) rotateX(${tiltY}deg) rotateY(${tiltX}deg)`;
      } else {
        // Reset de inclinación cuando el mouse sale del área expandida
        tiltBox.style.transform = '';
      }
    });
    const headerBar = document.querySelector('.header-bar');
    window.addEventListener('scroll', function () {
      if (window.scrollY > 0) {
        headerBar.classList.add('header-fixed');
      } else {
        headerBar.classList.remove('header-fixed');
      }
    });
    const toggleScheme = document.getElementById('toggle-scheme');
    if (toggleScheme) {
      const currentTheme = localStorage.getItem('theme') || 'light';
      document.documentElement.setAttribute('data-theme', currentTheme);
      toggleScheme.addEventListener('click', () => {
        const newTheme = document.documentElement.getAttribute('data-theme') === 'dark' ? 'light' : 'dark';
        document.documentElement.setAttribute('data-theme', newTheme);
        localStorage.setItem('theme', newTheme);
      });
    }

    // Función para inicializar el lienzo (canvas)
    function initCanvas(container) {
      const canvas = document.createElement('canvas');
      canvas.setAttribute('id', 'visualizerCanvas');
      canvas.setAttribute('class', 'visualizer-item');
      container.appendChild(canvas);
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
      return canvas;
    }

    // Función para cambiar el lienzo según el tamaño del contenedor
    function resizeCanvas(canvas, container) {
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
    }

    // Visualizer
    const visualizer = (audio, container) => {
      if (!audio || !container) {
        return;
      }
      const options = {
        fftSize: container.dataset.fftSize || 2048,
        numBars: container.dataset.bars || 80,
        maxHeight: 255
      };
      const ctx = new AudioContext();
      const audioSource = ctx.createMediaElementSource(audio);
      const analyzer = ctx.createAnalyser();
      audioSource.connect(analyzer);
      audioSource.connect(ctx.destination);
      const frequencyData = new Uint8Array(analyzer.frequencyBinCount);
      const canvas = initCanvas(container);
      const canvasCtx = canvas.getContext('2d');

      // Crear barras
      const renderBars = () => {
        resizeCanvas(canvas, container);
        analyzer.getByteFrequencyData(frequencyData);
        {
          analyzer.fftSize = options.fftSize;
        }
        canvasCtx.clearRect(0, 0, canvas.width, canvas.height);
        for (let i = 0; i < options.numBars; i++) {
          const index = Math.floor((i + 10) * (i < options.numBars / 2 ? 2 : 1));
          const fd = frequencyData[index];
          const barHeight = Math.max(4, fd / 255 * options.maxHeight);
          const barWidth = canvas.width / options.numBars;
          const x = i * barWidth;
          const y = canvas.height - barHeight;
          canvasCtx.fillStyle = 'white';
          canvasCtx.fillRect(x, y, barWidth - 2, barHeight);
        }
        requestAnimationFrame(renderBars);
      };
      renderBars();

      // Listener del cambio de espacio en la ventana
      window.addEventListener('resize', () => {
        resizeCanvas(canvas, container);
      });
    };

    /* eslint-disable camelcase */
    // Función para obtener el día actual en formato de 2 letras
    function getCurrentDay() {
      const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
      const date = new Date();
      const day = date.getDay();
      return days[day];
    }
    function compareNumbers(a, b) {
      return Number(a) === Number(b);
    }
    function sortByDay(programas) {
      const resultado = {
        monday: [],
        tuesday: [],
        wednesday: [],
        thursday: [],
        friday: [],
        saturday: [],
        sunday: []
      };
      programas.forEach(programa => {
        const {
          prog_horario_lu,
          prog_horario_ma,
          prog_horario_mi,
          prog_horario_ju,
          prog_horario_vi,
          prog_horario_sa,
          prog_horario_do,
          prog_lu,
          prog_ma,
          prog_mi,
          prog_ju,
          prog_vi,
          prog_sa,
          prog_do,
          prog_titulo,
          prog_descripcion,
          prog_foto
        } = programa;
        const item = {
          titulo: prog_titulo,
          descripcion: prog_descripcion,
          foto: prog_foto,
          horario: null
        };

        // Verificar cada día y agregar al arreglo correspondiente si está activo
        if (compareNumbers(prog_lu, 1)) {
          item.horario = prog_horario_lu;
          resultado.monday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ma, 1)) {
          item.horario = prog_horario_ma;
          resultado.tuesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_mi, 1)) {
          item.horario = prog_horario_mi;
          resultado.wednesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ju, 1)) {
          item.horario = prog_horario_ju;
          resultado.thursday.push({
            ...item
          });
        }
        if (compareNumbers(prog_vi, 1)) {
          item.horario = prog_horario_vi;
          resultado.friday.push({
            ...item
          });
        }
        if (compareNumbers(prog_sa, 1)) {
          item.horario = prog_horario_sa;
          resultado.saturday.push({
            ...item
          });
        }
        if (compareNumbers(prog_do, 1)) {
          item.horario = prog_horario_do;
          resultado.sunday.push({
            ...item
          });
        }
      });

      // Ordenar los programas por hora en cada día
      Object.keys(resultado).forEach(dia => {
        resultado[dia].sort((a, b) => a.horario.localeCompare(b.horario));
      });
      return resultado;
    }

    // Compara la hora actual con la hora de inicio de un programa
    // @param {string} time - Hora de inicio del programa
    // @returns {number} - 1 si la hora actual es mayor, -1 si la hora actual es menor, 0 si son iguales
    function compareTime(time) {
      const date = new Date();
      const currentHour = date.getHours();
      const currentMinutes = date.getMinutes();
      const [programHour, programMinutes] = time.split(':').map(Number);
      if (currentHour > programHour || currentHour === programHour && currentMinutes > programMinutes) {
        return 1; // Hora actual es mayor
      } else if (currentHour < programHour || currentHour === programHour && currentMinutes < programMinutes) {
        return -1; // Hora actual es menor
      } else {
        return 0; // Horas iguales
      }
    }
    function setupSchedule(data, baseUrl) {
      const programs = sortByDay(data);
      const days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
      const tabs = document.querySelectorAll('.program-button');
      const tabsElements = document.querySelectorAll('.nexo-schedule-list');
      tabs.forEach(tab => {
        tab.onclick = event => {
          const target = event.target;
          const tabId = target.dataset.tab;
          tabs.forEach(tab => {
            tab.classList.remove('is-active');
          });
          tabsElements.forEach(tab => {
            tab.classList.remove('is-active');
          });
          const ul = document.getElementById(tabId);
          ul.classList.add('is-active');
          target.classList.add('is-active');
        };
      });
      days.forEach(day => {
        const ul = document.getElementById(`program-${day}`);
        ul.innerHTML = '';
        programs[day].forEach(item => {
          const li = document.createElement('li');
          li.classList.add('nexo-schedule-item');
          li.innerHTML = `
        <div class="nexo-schedule-item-picture">
          <img src="${baseUrl}${item.foto}" alt="${item.titulo}">
        </div>
        <div class="nexo-schedule-item-content">
          <h3 class="nexo-schedule-item-title">${item.titulo}</h3>
          <p class="nexo-schedule-item-description">${item.horario} - ${item.descripcion}</p>
        </div>
      `;

          // Comparar la hora actual con la hora de inicio del programa
          const result = compareTime(item.horario);

          // Agregar la clase is-past si la hora actual es mayor
          // Solo agregar las clases si el programa es del día actual
          if (result > 0 && day === getCurrentDay()) {
            li.classList.add('is-past');
          }
          ul.appendChild(li);
        });

        // al ultimo elemento con la clase is-past agregarle la clase is-current
        // Solo si el día es el día actual
        const pastItems = document.querySelectorAll('.is-past');
        if (pastItems.length > 0 && day === getCurrentDay()) {
          pastItems[pastItems.length - 1].classList.add('is-current');
        }
      });

      // Mostrar el día actual
      const currentDay = getCurrentDay();

      // Mostrar el día actual
      const alltabs = document.querySelectorAll('.program-button');
      alltabs.forEach(tab => {
        tab.classList.remove('is-active');
      });
      const currentTab = document.querySelector(`[data-tab="program-${currentDay}"]`);
      currentTab.classList.add('is-active');
      const allLists = document.querySelectorAll('.nexo-schedule-list');
      allLists.forEach(list => {
        list.classList.remove('is-active');
      });
      const currentList = document.getElementById(`program-${currentDay}`);
      currentList.classList.add('is-active');
    }
    const icons = {
      volumeMute: '<i class="ri-volume-mute-fill"></i>',
      volumeUp: '<i class="ri-volume-up-fill"></i>'};
    const svgIcons = {
      play: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M10 16.5v-9l6 4.5M12 2A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>',
      pause: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M15 16h-2V8h2m-4 8H9V8h2m1-6A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>',
      facebook: '<svg class="i i-facebook" viewBox="0 0 24 24"><path d="M17 14h-3v8h-4v-8H7v-4h3V7a5 5 0 0 1 5-5h3v4h-3q-1 0-1 1v3h4Z"></path></svg>',
      youtube: '<svg class="i i-youtube" viewBox="0 0 24 24"><path d="M1.5 17q-1-5.5 0-10Q1.9 4.8 4 4.5q8-1 16 0 2.1.3 2.5 2.5 1 4.5 0 10-.4 2.2-2.5 2.5-8 1-16 0-2.1-.3-2.5-2.5Zm8-8.5v7l6-3.5Z"></path></svg>',
      instagram: '<svg class="i i-instagram" viewBox="0 0 24 24"><circle cx="12" cy="12" r="4"></circle><rect width="20" height="20" x="2" y="2" rx="5"></rect><path d="M17.5 6.5h0"></path></svg>',
      whatsapp: '<svg class="i i-whatsapp" viewBox="0 0 24 24"><circle cx="9" cy="9" r="1"></circle><circle cx="15" cy="15" r="1"></circle><path d="M8 9a7 7 0 0 0 7 7m-9 5.2A11 11 0 1 0 2.8 18L2 22Z"></path></svg>',
      tiktok: '<svg class="i i-tiktok" viewBox="0 0 24 24"><path d="M22 6v5q-4 0-6-2v7a7 7 0 1 1-5-6.7m0 6.7a2 2 0 1 0-2 2 2 2 0 0 0 2-2V1h5q2 5 6 5"></path></svg>',
      bluesky: '<svg class="i i-bluesky" viewBox="0 0 24 24"><path d="M12 10Q2-2 2 6t5 8q-5 3-1 6t6-3q2 6 6 3t-1-6q5 0 5-8t-10 4"></path></svg>',
      discord: '<svg class="i i-discord" viewBox="0 0 24 24"><path d="M9 3q-2.5.5-5 2-3 5-3 12 2 2.5 6 4 1-1.5 1.5-3.5M7 17q5 2 10 0m-1.5.5q.5 2 1.5 3.5 4-1.5 6-4 0-7-3-12-2.5-1.5-5-2l-1 2q-2-.5-4 0L9 3"></path><circle cx="8" cy="12" r="1"></circle><circle cx="16" cy="12" r="1"></circle></svg>',
      soundcloud: '<svg class="i i-soundcloud" viewBox="0 0 24 24"><path d="M11 7v11m-3 0v-7m-6 6v-3m3-4v8m9 0h6a1 1 0 0 0 0-6 6 6 0 0 0-6-6Z"></path></svg>',
      threads: '<svg class="i i-threads" viewBox="0 0 24 24"><path d="M20 8C18 0 6 0 4 8s1.5 14 8 14 10-7.6 4-10-9 2-7 4 7 1 7-4-5-6-7-4"></path></svg>',
      twitter: '<svg class="i i-twitter" viewBox="0 0 24 24"><path d="M12 7.5a4.5 4.5 0 0 1 8-3Q22 4 23 3q0 2-2 4A13 13 0 0 1 1 19q5 0 7-2-8-4-5-13 4 5 9 5Z"></path></svg>',
      x: '<svg class="i i-x" viewBox="0 0 24 24"><path d="m3 21 7.5-7.5m3-3L21 3M8 3H3l13 18h5Z"></path></svg>',
      deezer: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" d="M20.1 3.6c.2-1.2.6-2 1-2 .6 0 1.1 2.8 1.1 6.2 0 3.5-.5 6.3-1.2 6.3-.3 0-.5-.5-.7-1.3-.3 3-1 5-1.8 5-.6 0-1.1-1.3-1.5-3.2-.3 3.7-.9 6.2-1.6 6.2-.5 0-.9-1-1.2-2.7C13.8 21.6 13 24 12 24c-1 0-1.9-2.3-2.3-5.8-.3 1.7-.7 2.7-1.1 2.7-.8 0-1.4-2.6-1.6-6.2-.4 2-1 3.1-1.5 3.1-.8 0-1.5-2-1.8-4.9-.2.8-.5 1.3-.8 1.3-.6 0-1.2-2.8-1.2-6.3 0-3.4.6-6.2 1.2-6.2.4 0 .7.8 1 2C4.1 1.5 4.8 0 5.4 0c.7 0 1.4 2 1.7 5 .4-2.2.8-3.5 1.4-3.5.7 0 1.4 2.7 1.6 6.4.4-1.9 1-3.1 1.8-3.1.7 0 1.4 1.2 1.8 3.1.2-3.7.9-6.4 1.6-6.4.5 0 1 1.3 1.3 3.5.3-3 1-5 1.8-5 .7 0 1.2 1.4 1.6 3.6zM.7 10C.3 10 0 8.7 0 7.2c0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8S1.1 10 .7 10zm22.6 0c-.4 0-.7-1.3-.7-2.8 0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8s-.3 2.8-.7 2.8z"/></svg>'
    };
    function getSocials(data) {
      const socials = {};
      if (data.radio_facebook_url) {
        socials.facebook = data.radio_facebook_url;
      }
      if (data.radio_youtube) {
        socials.youtube = data.radio_youtube;
      }
      if (data.radio_instagram) {
        socials.instagram = data.radio_instagram;
      }
      if (data.radio_whatsapp) {
        socials.whatsapp = `https://wa.me/+${data.radio_whatsapp}`;
      }
      if (data.radio_menu_noticias) {
        socials.tiktok = data.radio_menu_noticias;
      }
      return socials;
    }

    // Convertir a tiempo relativo
    function timeAgo(date) {
      const getSecondsDiff = timestamp => (Date.now() - timestamp) / 1000;

      // Unidades de tiempo
      const DATE_UNITS = {
        day: 86400,
        hour: 3600,
        minute: 60,
        second: 1
      };

      // Obtener unidad y valor de la fecha
      const getUnitAndValueDate = secondsElapsed => {
        for (const [unit, secondsInUnit] of Object.entries(DATE_UNITS)) {
          if (secondsElapsed >= secondsInUnit || unit === 'second') {
            const value = Math.floor(secondsElapsed / secondsInUnit) * -1;
            return {
              value,
              unit
            };
          }
        }
      };

      // Obtener tiempo relativo
      const getTimeAgo = timestamp => {
        const rtf = new Intl.RelativeTimeFormat();
        const secondsElapsed = getSecondsDiff(timestamp);
        const {
          value,
          unit
        } = getUnitAndValueDate(secondsElapsed);
        return rtf.format(value, unit);
      };

      // Resultado
      const reference = new Date(date);
      return getTimeAgo(reference);
    }

    // Normaliza los datos de la estación
    // @param {object} data - Datos de la estación
    // @param {string} baseUrl - URL base de la API
    function getStationData(data, baseUrl, isMultiStream = false) {
      const station = {
        name: isMultiStream ? data.multiradio_nombre : data.radio_nombre,
        description: isMultiStream ? data.multiradio_descripcion : data.radio_descripcion || 'Sonando En Vivo',
        stream_url: isMultiStream ? data.multiradio_url : data.radio_url,
        tv_url: isMultiStream ? data.multiradio_tvstream : data.radio_video,
        picture: isMultiStream ? `${baseUrl}${data.multiradio_imagen}` : `${baseUrl}${data.radio_fondo}`,
        background: isMultiStream ? `${baseUrl}${data.multiradio_imagen}` : `${baseUrl}${data.radio_splash}`,
        api: isMultiStream ? data.multiradio_api : data.radio_menu_nicio
      };
      return station;
    }

    // Devolver una promesa para saber si la imagen se ha cargado correctamente
    // @param {string} src - URL de la imagen
    // @returns {Promise} - Promesa que se resuelve si la imagen se carga correctamente
    function loadImage(src) {
      return new Promise((resolve, reject) => {
        const img = new Image();
        img.onload = () => resolve(img);
        img.onerror = reject;
        img.src = src;
      });
    }

    // Agrega una transición de deslizamiento a la imagen
    // @param {HTMLElement} container - Contenedor de la imagen
    // @param {string} src - URL de la imagen
    function slideUpImageTransition(container, src) {
      const img = document.createElement('img');
      const size = container.clientHeight;
      img.src = src;
      img.width = size;
      img.height = size;
      container.appendChild(img);
      const firstImg = container.querySelector('img:first-child');
      firstImg.style.marginLeft = `-${size}px`;
      firstImg.addEventListener('transitionend', () => {
        const allImgExcLast = container.querySelectorAll('img:not(:last-child)');
        allImgExcLast.forEach(img => img.remove());
      }, {
        once: true
      });
    }

    // Agrega una transición de desvanecimiento a la imagen
    // @param {HTMLElement} container - Contenedor de la imagen
    // @param {string} src - URL de la imagen
    function fadeImageTransition(container, src) {
      container.style.opacity = 0;
      container.addEventListener('transitionend', () => {
        container.src = src;
        container.style.opacity = 0.2;
        container.addEventListener('transitionend', () => {
          container.removeAttribute('style');
        }, {
          once: true
        });
      });
    }

    // Crear una imagen proxy
    // @param {string} src - URL de la imagen
    // @returns {Promise} - Promesa que se resuelve con la imagen
    function createProxyImage(src) {
      return new Promise((resolve, reject) => {
        const img = document.createElement('img');
        img.crossOrigin = 'Anonymous';
        img.src = `https://images.weserv.nl/?url=${src}`;
        img.onload = () => resolve(img);
        img.onerror = reject;
      });
    }

    // Establecer el color de acento de la página
    // @param {HTMLImageElement} image - Imagen a extraer el color
    // @param {ColorThief} colorThief - Instancia de ColorThief
    function setAccentColor(image, colorThief) {
      const metaThemeColor = document.querySelector('meta[name=theme-color]');

      // Función para establecer el color
      const applyColor = () => {
        const color = `rgb(${colorThief.getColor(image)})`;
        document.querySelector('html').style.setProperty('--player-accent', color);
        if (metaThemeColor) {
          metaThemeColor.setAttribute('content', color);
        }
      };
      if (image.complete) {
        applyColor();
      } else {
        image.addEventListener('load', applyColor);
      }
    }

    // Set attributes
    // @param {HTMLElement} element - element
    // @param {Object} attributes - attributes
    // @returns {void}
    const setAttributes = (element, attributes) => {
      for (const key in attributes) {
        element.setAttribute(key, attributes[key]);
      }
    };

    // Create embed
    // @param {string} url - url
    // @param {Object} params - attributes
    // @returns {HTMLIFrameElement} - iframe element
    function createEmbed({
      url,
      params = {}
    }) {
      const $iframe = document.createElement('iframe');
      $iframe.src = url;
      $iframe.frameborder = 0;
      setAttributes($iframe, params);
      return $iframe;
    }

    // Get video info
    // @param {string} url - url
    // @returns {Promise<string>} - title
    async function getVideoInfo(videoId) {
      const data = await fetch(`https://noembed.com/embed?dataType=json&url=https://www.youtube.com/watch?v=${videoId}`);
      return data.json();
    }
    const modal = document.querySelector('.modal');
    const modalBody = modal.querySelector('.modal-body');
    const OPEN_CLASS = 'is-open';
    let currentVideo = null;
    function initVideoPlayer(videoUrl) {
      const $video = document.createElement('video');
      $video.id = 'player';
      $video.classList.add('video-js', 'video-js-stream');
      modalBody.innerHTML = '';
      modalBody.appendChild($video);
      const videoId = document.getElementById('player');

      // eslint-disable-next-line no-undef
      currentVideo = videojs(videoId, {
        sources: [{
          src: videoUrl,
          type: 'application/x-mpegURL'
        }],
        controls: true,
        autoplay: true
      });
    }
    function createShareButton(name, url) {
      const $a = document.createElement('a');
      $a.className = `player-social-link player-social-link-${name}`;
      $a.href = url;
      $a.target = '_blank';
      $a.innerHTML = svgIcons[name];
      return $a;
    }
    function initShareModal({
      artist,
      title,
      artwork
    }) {
      const shareUrl = window.location.href;
      const shareText = `Estoy escuchando ${title} de ${artist} en ${shareUrl}`;
      modalBody.innerHTML = `
    <h2 class="fw-800 mb-0.75">COMPARTIR</h2>
    <img src="${artwork}" alt="portada modal" class="player-modal-image" width="200" height="200">
    <div class="fs-4 fw-500 song-name">${title}</div>
    <span class="fs-5">${artist}</span>
    <div class="flex g-0.5 justify-center modal-social player-socials wrap">
      ${createShareButton('facebook', `https://www.facebook.com/sharer/sharer.php?u=${shareUrl}`).outerHTML}
      ${createShareButton('x', `https://twitter.com/intent/tweet?text=${encodeURIComponent(shareText)}`).outerHTML}
      ${createShareButton('whatsapp', `https://api.whatsapp.com/send?text=${encodeURIComponent(shareText)}`).outerHTML}
    </div>
  `;
    }
    function createForm(formAction, formId) {
      modalBody.innerHTML = `
    <form id="${formId}" action="${formAction}" method="POST" class="modal-form flex column g-1">
      <input type="text" name="name" placeholder="Nombre" required>
      <input type="email" name="email" placeholder="Correo" required>
      <input type="tel" name="contact" placeholder="Teléfono" required>
      <textarea name="message" placeholder="Mensaje" required></textarea>
      <input type="hidden" name="promo" value="${formId}">
      <button type="submit" class="btn btn-full btn-input">Enviar</button>
    </form>
  `;
      const $form = modalBody.querySelector('form');
      $form.addEventListener('submit', async e => {
        e.preventDefault();
        const formData = new FormData($form);
        const response = await fetch(formAction, {
          method: 'POST',
          body: formData
        });
        document.body.classList.add(response.ok ? 'modal-success' : 'modal-error');
        setTimeout(() => document.body.classList.remove('modal-success', 'modal-error'), 3000);
        if (response.ok) $form.reset();
      });
    }
    function openLyricsModal(lyrics) {
      modalBody.innerHTML = `
    <div class="lyrics-header">
      <h2 class="modal-lyrics-title m:fs-3 fs-5 fw-500">Letra</h2>
    </div>
    <pre>${lyrics || 'No se encontraron letras'}</pre>
  `;
    }
    function openSongModal({
      artist,
      title,
      artwork,
      genre,
      album
    }) {
      modalBody.innerHTML = `<div class="modal-song-info p-1 text-center">
    <img src="${artwork}" alt="portada modal" class="player-modal-image" width="200" height="200">
    <h3 class="fs-4 fw-700 color-text-inverse song-name">${title}</h3>
    <div class="flex column color-text-inverse justify-center modal-social player-socials wrap">
      <span class="fs-5"><strong class="color-text-inverse fw-700">Artista: </strong>${artist}</span>
      <span class="fs-5"><strong class="color-text-inverse fw-700">Genero: </strong>${genre}</span>
      <span class="fs-5"><strong class="color-text-inverse fw-700">Album: </strong>${album}</span>
    </div>
  </div>
  `;
    }
    function openFullModal({
      titulo,
      contenido,
      imagen,
      time
    }) {
      document.body.classList.add('modal-article');
      modalBody.innerHTML = `
    <div class="modal-article-title">
      <h2 class="fs-5 fw-800 m:fs-2">${titulo}</h2>
      <span class="fs-5">${time}</span>
    </div>
    <div class="modal-article-content">
      ${contenido}
    </div>
  `;
    }
    function openModal(type, options = {}) {
      if (!modal) return;
      modal.classList.add(OPEN_CLASS);
      modalBody.innerHTML = '';
      modal.classList.add(`modal-${type}`);
      switch (type) {
        case 'video':
          initVideoPlayer(options.url);
          break;
        case 'share':
          initShareModal(options.data);
          break;
        case 'form':
          createForm(options.formAction, options.formId);
          break;
        case 'lyrics':
          openLyricsModal(options.lyrics);
          break;
        case 'song':
          openSongModal(options.data);
          break;
        case 'article':
          openFullModal(options.data);
          break;
        default:
          console.warn('Tipo de modal desconocido:', type);
          return;
      }
      function closeModal() {
        modal.classList.remove(OPEN_CLASS, `modal-${type}`);
        document.body.classList.remove('modal-article');
        if (currentVideo) currentVideo.dispose();
        modalBody.innerHTML = '';
        if (options.hook && typeof options.hook === 'function') {
          options.hook();
        }
      }
      modal.querySelector('.modal-close').addEventListener('click', closeModal);
      modal.addEventListener('click', e => {
        if (!modal.querySelector('.modal-content').contains(e.target)) closeModal();
      });
    }
    let currentIframeActive = null;
    const cache$1 = localStorage.getItem('ytListMini') ? JSON.parse(localStorage.getItem('ytListMini')) : {};
    async function fetchVideoList(data, CONTAINER_LIST) {
      // max 4 videos
      const items = data.length > 5 ? data.slice(0, 5) : data;
      for (const item of items) {
        const videoInfo = cache$1[item.video_url] || (await getVideoInfo(item.video_url));
        const $li = createVideoListItem(item.video_url, videoInfo);
        CONTAINER_LIST.appendChild($li);
        addEventListeners($li, item.video_url);
        cache$1[item.video_url] = videoInfo;
        localStorage.setItem('ytListMini', JSON.stringify(cache$1));
      }
      const $firstVideo = CONTAINER_LIST.querySelector('.ytlistmini-load-iframe');
      if ($firstVideo) {
        $firstVideo.click();
      }
    }
    function createVideoListItem(videoId, videoInfo) {
      const $div = document.createElement('div');
      const videoThumbnail = videoInfo.thumbnail_url.replace('hqdefault', 'mqdefault');
      $div.className = 'ytlistmini-item';
      $div.innerHTML = `
    <div class="ytlistmini-content" data-video-id="${videoId}">
      <div class="ytlistmini-element">
        <img src="${videoThumbnail}" alt="${videoInfo.title}">
        <div class="ytlistmini-info">
          <h3>${videoInfo.title}</h3>
          <p>${videoInfo.author_name}</p>
        </div>
        <button class="ytlistmini-load-iframe"></button>
      </div>
      <div class="ytlistmini-iframe"></div>
    </div>
  `;
      return $div;
    }
    function addEventListeners($li, videoId) {
      $li.querySelector('.ytlistmini-load-iframe').addEventListener('click', function () {
        handleLoadIframeButtonClick($li, videoId);
      });
    }
    function handleLoadIframeButtonClick($li, videoId) {
      const $iframe = createEmbed({
        url: `https://www.youtube.com/embed/${videoId}`,
        params: {
          allowfullscreen: '1',
          class: 'ytube-embed'
        }
      });
      if ($li.classList.contains('is-active')) return;
      $li.classList.add('is-active');
      if (currentIframeActive) {
        currentIframeActive.innerHTML = '';
        currentIframeActive.classList.remove('is-active');
        currentIframeActive.parentElement.parentElement.classList.remove('is-active');
      }
      const $iframeContainer = $li.querySelector('.ytlistmini-iframe');
      $iframeContainer.innerHTML = '';
      $iframeContainer.appendChild($iframe);
      currentIframeActive = $iframeContainer;
    }
    const TIMEOUT$1 = window?.streams?.sliderTiming || 5000;

    // @param {Object} data - Datos de la estación
    function setupStream(container, data) {
      if (!container || !data) return;
      const pictureEl = container.querySelector('.player-picture img:first-child');
      const backgroundEl = container.querySelector('.player-cover');
      const stationName = container.querySelector('.station-name');
      const stationDescription = container.querySelector('.station-description');
      const stationTv = container.querySelector('.station-tv');
      if (pictureEl) {
        pictureEl.src = data.picture;
      }
      if (backgroundEl) {
        backgroundEl.src = data.background;
      }
      if (stationName) {
        stationName.textContent = data.name;
      }
      if (stationDescription) {
        stationDescription.textContent = data.description;
      }
      if (stationTv) {
        stationTv.classList.remove('none');
      } else {
        stationTv.classList.add('none');
      }
    }

    // Establecer el slider
    // @param {HTMLElement} container - Contenedor del slider
    // @param {Array} data - Datos de las imágenes
    // @param {string} baseUrl - URL base de las imágenes
    function setupSlider(container, data, baseUrl) {
      const banners = data.map(item => {
        return `<div class="slider-slide">
      <a href="${item.slide_url}" target="_blank">
        <img src="${baseUrl}${item.slide_foto}" alt="Slide" class="slider-image" width="100%">
      </a>
    </div>`;
      });
      container.innerHTML = banners.join('');
      container.classList.add('is-loaded');
      if (container) {
        const SLIDES = data.length;

        // Mover el contenedor Wrapper dependiendo de la cantidad de slides
        let currentSlide = 0;
        let intervalId;
        function moveSlide() {
          if (currentSlide < SLIDES - 1) {
            container.style.transform = `translateX(-${(currentSlide + 1) * 100}%)`;
            currentSlide++;
          } else {
            container.style.transform = 'translateX(0)';
            currentSlide = 0;
          }
          intervalId = setTimeout(moveSlide, TIMEOUT$1);
        }
        function startSlider() {
          intervalId = setTimeout(moveSlide, TIMEOUT$1);
        }
        function stopSlider() {
          clearTimeout(intervalId);
        }
        container.addEventListener('mouseover', stopSlider);
        container.addEventListener('mouseout', startSlider);
        container.addEventListener('mousedown', stopSlider);
        container.addEventListener('mouseup', startSlider);
        setTimeout(moveSlide, TIMEOUT$1);
      }
    }

    // Establecer las redes sociales
    // @param {HTMLElement} container - Contenedor de las redes sociales
    // @param {object} data - Datos de las redes sociales
    function setupSocial(container, data) {
      if (!container || !data) return;
      const socialItems = Object.keys(data).map(key => {
        return `<a href="${data[key]}" target="_blank" class="player-social-link player-social-link-${key}">
      ${svgIcons[key]}
    </a>`;
      });
      container.innerHTML = socialItems.join('');
    }

    // Obtener los datos de la búsqueda
    // @param {string} query - Consulta de búsqueda
    // @param {string} service - Servicio de búsqueda
    // @returns {Promise<object>} - Datos de la búsqueda
    const cache = {};
    const getDataFromSearchApi = async (query, service) => {
      if (cache[query]) {
        return cache[query];
      }
      const streamUrl = `api/search.php?query=${encodeURIComponent(query)}&service=${service}`;
      const response = await fetch(streamUrl);
      const data = await response.json();

      // Si no responde
      if (!data.results) {
        return {};
      }
      const results = data.results;
      cache[query] = results;
      return results;
    };

    // Obtener datos del stream
    // @param {string} streamUrl - URL del stream
    // @returns {Promise<object>} - Datos del stream
    async function getStreamData(streamUrl) {
      if (!streamUrl) {
        return {};
      }

     
      const jsonUri = `api/met?url=${encodeURIComponent(streamUrl)}`;
    

      try {
        const response = await fetch(jsonUri);
        return response.json();
      } catch (error) {
        console.error('Error al obtener los datos del stream:', error);
        return {};
      }
    }

    // Obtener letras de canciones usando Lyrics.ovh
    const getLyricsOvh = async (artist, name) => {
      try {
        const response = await fetch(`https://api.lyrics.ovh/v1/${encodeURIComponent(artist)}/${encodeURIComponent(name)}?auto=1`);
        if (!response.ok) throw new Error('No se pudo obtener la letra');
        const data = await response.json();
        if (data.lyrics) {
          return data.lyrics;
        } else {
          return 'Letra no disponible';
        }
      } catch (error) {
        console.warn('No se pudo obtener la letra de Lyrics.ovh');
        return 'Letra no disponible';
      }
    };

    // obtener letras de vagalume
    const getVagalumeLyrics = async (artist, name) => {
      const apikey = '1637b78dc3b129e6843ed674489a92d0';
      try {
        const response = await fetch(`https://api.vagalume.com.br/search.php?apikey=${apikey}&art=${encodeURIComponent(artist)}&mus=${encodeURIComponent(name)}`);
        if (!response.ok) throw new Error('No se pudo obtener la letra');
        const data = await response.json();
        if (data.type === 'notfound') {
          return 'Letra no disponible';
        }
        return data.mus[0].text;
      } catch (error) {
        console.warn('No se pudo obtener la letra de Vagalume');
        return 'Letra no disponible';
      }
    };

    // obtener letras
    const getLyrics = async (artist, name) => {
      let lyrics = await getLyricsOvh(artist, name);
      if (!lyrics || lyrics === 'Letra no disponible') {
        lyrics = await getVagalumeLyrics(artist, name);
      }
      return lyrics;
    };
    const player = document.querySelector('.player');
    const playerSocial = document.querySelector('.player-social');
    const playButton = document.querySelector('.player-button-play');
    const visualizerContainer = document.querySelector('.visualizer');
    const isWave = localStorage.getItem('isWave') === 'true' || false;
    const rhythm = document.querySelector('.player-button-rhythm-toggle');

    // --- NUEVA VARIABLE DE TOKEN ---
    // Este token se incrementa cada vez que se inicia una nueva actualización de estación.
    let currentUpdateToken = 0;
    const STREAMS = window?.streams || {};
    const TIMEOUT = STREAMS?.timeRefresh || 5000;
    const SERVICE = STREAMS?.service || 'spotify';
    const MODULE_VIDEO_TOPS = STREAMS?.module_video_tops;
    const MODULE_NEWS = STREAMS?.module_news;
    const MODULE_TEAM = STREAMS?.module_team;
    const CAROUSEL_TIME = STREAMS?.carouselTiming || 5000;
    if (isWave) {
      visualizerContainer.classList.add('is-wave');
      rhythm.classList.add('is-wave');
    }
    rhythm.addEventListener('click', () => {
      rhythm.classList.toggle('is-wave');
      visualizerContainer.classList.toggle('is-wave');
      localStorage.setItem('isWave', visualizerContainer.classList.contains('is-wave'));
    });
    const audio = new Audio();
    let hasVisualizer = false;
    audio.crossOrigin = 'anonymous';
    if (playButton !== null) {
      playButton.addEventListener('click', async () => {
        if (audio.paused) {
          play(playButton);
        } else {
          pause(playButton);
        }
      });
    }
    function play(button, newSource = null) {
      if (newSource) {
        audio.src = newSource;
      }
      audio.load();
      audio.play();
      button.innerHTML = `${svgIcons.pause} Pausar`;
      button.classList.add('is-active');
      document.body.classList.add('is-playing');

      // Visualizer
      if (!hasVisualizer) {
        visualizer(audio, visualizerContainer);
        hasVisualizer = true;
      }
    }
    function pause(button) {
      audio.pause();
      button.innerHTML = `${svgIcons.play} Reproducir`;
      button.classList.remove('is-active');
      document.body.classList.remove('is-playing');
    }

    // Botón de volumen
    const range = document.querySelector('.player-volume');
    const rangeFill = document.querySelector('.player-range-fill');
    const rangeWrapper = document.querySelector('.player-range-wrapper');
    const rangeThumb = document.querySelector('.player-range-thumb');
    const volumeToggle = document.querySelector('.player-button-volume-toggle');
    const currentVolume = localStorage.getItem('volume') || 100;

    // Rango recorrido
    function setRangeWidth(percent) {
      {
        rangeFill.style.height = `${percent}%`;
      }
    }

    // Posición del thumb
    function setThumbPosition(percent) {
      const compensatedWidth = rangeWrapper.offsetHeight - rangeThumb.offsetWidth;
      const thumbPosition = percent / 100 * compensatedWidth;
      {
        rangeThumb.style.bottom = `${thumbPosition}px`;
      }
    }

    // Actualiza el volumen al cambiar el rango
    function updateVolume(value) {
      range.value = value;
      setRangeWidth(value);
      setThumbPosition(value);
      localStorage.setItem('volume', value);
      audio.volume = value / 100;
      if (value === 0) {
        volumeToggle.innerHTML = icons.volumeMute;
      } else {
        volumeToggle.innerHTML = icons.volumeUp;
      }
    }

    // Valor inicial
    if (range !== null) {
      updateVolume(currentVolume);

      // Escucha el cambio del rango
      range.addEventListener('input', event => {
        updateVolume(event.target.value);
      });

      // Escucha el movimiento del mouse
      rangeThumb.addEventListener('mousedown', () => {
        document.addEventListener('mousemove', handleThumbDrag);
      });
    }

    // Intercambiar entre 0% y 100% de volumen
    if (volumeToggle !== null) {
      volumeToggle.addEventListener('click', () => {
        if (audio.volume > 0) {
          updateVolume(0);
        } else {
          updateVolume(100);
        }
      });
    }

    // Cargar datos de la canción actual al navegador
    function setMediaSession(data, picture, token) {
      // Si en algún momento el token no coincide, se descarta esta actualización.
      if (token !== currentUpdateToken) return;
      const {
        artist,
        artwork
      } = data;
      const title = data.title || data.song;
      if ('mediaSession' in navigator) {
        navigator.mediaSession.metadata = new MediaMetadata({
          title,
          artist,
          artwork: [{
            src: artwork || picture,
            sizes: '128x128',
            type: 'image/png'
          }, {
            src: artwork || picture,
            sizes: '256x256',
            type: 'image/png'
          }, {
            src: artwork || picture,
            sizes: '512x512',
            type: 'image/png'
          }]
        });
        navigator.mediaSession.setActionHandler('play', () => {
          play();
        });
        navigator.mediaSession.setActionHandler('pause', () => {
          pause();
        });
      }
    }

    // Mueve el thumb y actualiza el volumen
    function handleThumbDrag(event) {
      const rangeRect = range.getBoundingClientRect();
      const clickX = event.clientY - rangeRect.top;
      let percent = clickX / range.offsetWidth * 100;
      percent = 100 - percent; // Invertir el porcentaje si es vertical

      percent = Math.max(0, Math.min(100, percent));
      const value = Math.round((range.max - range.min) * (percent / 100)) + parseInt(range.min);
      updateVolume(value);
    }

    // Deja de escuchar el movimiento del mouse
    document.addEventListener('mouseup', () => {
      document.removeEventListener('mousemove', handleThumbDrag);
    });
    window.addEventListener('resize', () => {
      const currentPercent = range.value;
      setRangeWidth(currentPercent);
      setThumbPosition(currentPercent);
    });

    // Establecer datos de la canción actual
    function setCurrentSong(data, picture, token) {
      // Si en algún momento el token no coincide, se descarta esta actualización.
      if (token !== currentUpdateToken) return;
      const {
        artist,
        artwork
      } = data;
      const title = data.title || data.song;
      const songName = document.querySelector('.song-name');
      const songArtist = document.querySelector('.song-artist');
      const pageTitle = document.querySelector('title');
      if (songName) {
        songName.textContent = title;
      }
      if (songArtist) {
        songArtist.textContent = artist;
      }
      if (pageTitle) {
        pageTitle.textContent = `${title} - ${artist}`;
      }
      const pictureEl = document.querySelector('.player-picture');
      const artworkUrl = artwork || picture;

      // eslint-disable-next-line no-undef
      const colorThief = new ColorThief();

      // Establecer el color de acento
      createProxyImage(artworkUrl).then(img => {
        setAccentColor(img, colorThief);
      }).catch(() => {
        console.error('Error al cargar la imagen');
      });
      if (pictureEl) {
        loadImage(artworkUrl).then(() => {
          // Verifica nuevamente el token antes de aplicar la transición
          if (token === currentUpdateToken) {
            slideUpImageTransition(pictureEl, artworkUrl);
          }
        }).catch(() => {
          console.error('Error al cargar arte de la canción');
        });
      }
      const coverEl = document.querySelector('.player-cover');
      if (coverEl) {
        const coverUrl = artwork || picture;
        const $img = document.createElement('img');
        $img.src = coverUrl;
        loadImage(coverUrl).then(() => {
          if (token === currentUpdateToken) {
            fadeImageTransition(coverEl, coverUrl);
          }
        }).catch(() => {
          console.error('Error al cargar la portada de la canción');
        });
      }
    }

    // Establecer las canciones que se han reproducido
    async function setHistory(data, current, token) {
      // Si en algún momento el token no coincide, se descarta esta actualización.
      if (token !== currentUpdateToken) return;
      const historyContainer = document.querySelector('.last-played');
      if (!historyContainer || !data) return;
      const historyItems = historyContainer.querySelectorAll('.history').length;
      data = data.slice(0, historyItems);

      // Array para almacenar los datos de las canciones
      const historyCurrentData = [];
      const itemsHTML = await Promise.all(data.map(async ({
        song: {
          artist,
          title
        }
      }, index) => {
        const dataFrom = await getDataFromSearchApi(`${artist} - ${title}`, SERVICE);
        if (dataFrom.artwork.includes('https://radioapi.lat/') || dataFrom.artwork === null || title.trim() === '') {
          dataFrom.artwork = current.picture || current.cover;
          dataFrom.title = current.name;
          dataFrom.artist = current.description;
        } else {
          dataFrom.title = title;
          dataFrom.artist = artist;
        }

        // Almacenar los datos en el índice correcto
        historyCurrentData[index] = dataFrom;
        return `
        <div class="history" data-index="${index}">
          <div class="history-wrapper">
            <img class="history-art" src="${dataFrom.artwork}" alt="${dataFrom.title}">
            <div class="history-meta fs-7">
              <span class="history-title fw-700 uppercase">${dataFrom.title}</span>
              <span class="history-artist">${dataFrom.artist}</span>
            </div>
          </div>
          <a class="history-url history-deezer" href="${dataFrom.stream || '#not-found'}" target="_blank">
            ${svgIcons.deezer}
          </a>
          <button class="history-button"></button>
        </div>`;
      }));
      if (token === currentUpdateToken) {
        historyContainer.innerHTML = itemsHTML.join('');
        const historyItemsElements = historyContainer.querySelectorAll('.history');
        historyItemsElements.forEach(item => {
          const button = item.querySelector('.history-button');
          const index = item.getAttribute('data-index');
          button.addEventListener('click', event => {
            // Abrir modal con la canción seleccionada
            event.preventDefault();
            openModal('song', {
              data: historyCurrentData[index]
            });
          });
        });
      }
    }
    function setArticles(data, baseUrl) {
      const newsContainer = document.getElementById('news');
      if (!newsContainer || !data) return;
      data = data.slice(0, 8);
      for (const item of data) {
        const {
          titulo,
          photo,
          contenido
        } = item;
        const time = new Date(item.created_at);
        const createdAt = timeAgo(time);
        const article = document.createElement('article');
        article.classList.add('news-item', 'swiper-slide');
        article.innerHTML = `
      <div class="news-image">
        <img src="${baseUrl}${photo}" alt="${titulo}">
      </div>
      <div class="news-content">
        <h3 class="news-title">${titulo}</h3>
        <time class="news-date">${createdAt}</time>
      </div>
    `;
        article.addEventListener('click', () => {
          openModal('article', {
            data: {
              titulo,
              contenido,
              imagen: `${baseUrl}${photo}`,
              time: createdAt
            }
          });
        });
        newsContainer.appendChild(article);
      }

      // eslint-disable-next-line no-undef, no-unused-vars
      new Swiper('.swiper-news', {
        // Enabled autoplay mode
        autoplay: {
          delay: CAROUSEL_TIME,
          disableOnInteraction: false
        },
        // If we need navigation
        navigation: {
          nextEl: '.swiper-next-news',
          prevEl: '.swiper-prev-news'
        },
        // Responsive breakpoints
        breakpoints: {
          480: {
            slidesPerView: 2,
            spaceBetween: 12
          },
          640: {
            slidesPerView: 3,
            spaceBetween: 12
          },
          1024: {
            slidesPerView: 4,
            spaceBetween: 12
          }
        }
      });
    }
    function setTeam(data, baseUrl) {
      const teamContainer = document.getElementById('team');
      if (!teamContainer || !data) return;
      data = data.slice(0, 8);
      for (const item of data) {
        const equipoNombre = item.equipo_nombre;
        const equipoFoto = item.equipo_foto;
        const equipoDescripcion = item.equipo_descripcion;
        const article = document.createElement('article');
        article.classList.add('team-item', 'swiper-slide');
        article.innerHTML = `
      <div class="team-image">
        <img src="${baseUrl}${equipoFoto}" alt="${equipoNombre}">
      </div>
      <div class="team-content">
        <h3 class="team-title">${equipoNombre}</h3>
        <p class="team-description">${equipoDescripcion}</p>
      </div>
    `;
        teamContainer.appendChild(article);
      }

      // eslint-disable-next-line no-undef, no-unused-vars
      new Swiper('.swiper-team', {
        // Enabled autoplay mode
        autoplay: {
          delay: CAROUSEL_TIME,
          disableOnInteraction: false
        },
        // If we need navigation
        navigation: {
          nextEl: '.swiper-next-team',
          prevEl: '.swiper-prev-team'
        },
        // Responsive breakpoints
        breakpoints: {
          480: {
            slidesPerView: 2,
            spaceBetween: 12
          },
          640: {
            slidesPerView: 3,
            spaceBetween: 12
          },
          1024: {
            slidesPerView: 4,
            spaceBetween: 12
          }
        }
      });
    }

    // ----------------
    // Iniciar Datos
    let timeoutId;
    let timeoutIdProgram;
    let currentSongPlaying = 'none';
    let loadStations = [];
    let currentStation;
    let currentActiveStation;
    const json = window.streams || {};
    const baseUrl = json.base_url || '';
    const urlServer = json.url_server || 'api/';
    const idUser = json.id_user || '';
    const isMultiStream = json.multi_stream || false;
    const stationsContainer = document.getElementById('player-prm');
    const stationsButton = document.querySelector('.station-prm');
    const radioPath = isMultiStream ? 'multiradio-app/' : 'radio-app/';
    const dataUrl = `${baseUrl}${urlServer}${radioPath}${idUser}`;
    const dataSocialUrl = `${baseUrl}${urlServer}radio-app/${idUser}`;
    const dataProgramUrl = `${baseUrl}${urlServer}programa-app/${idUser}`;
    const dataBannersUrl = `${baseUrl}${urlServer}slide-app/${idUser}`;
    const dataPromoUrl = `${baseUrl}${urlServer}promocion-app/${idUser}`;
    const dataGetPromoUrl = `${baseUrl}${urlServer}promotional-contact`;
    const dataNoticiasUrl = `${baseUrl}${urlServer}noticia-app/${idUser}`;
    const dataVideoUrl = `${baseUrl}${urlServer}video-app/${idUser}`;
    const dataTeamUrl = `${baseUrl}${urlServer}equipo-app/${idUser}`;
    const stationTv = document.querySelector('.station-tv');
    function setupAll(station) {
      audio.src = station.stream_url;
      setupStream(player, station);
      if (station.tv_url) {
        stationTv.onclick = () => {
          openModal('video', {
            url: station.tv_url,
            hook: () => play(playButton)
          });
          pause(playButton);
        };
      }
    }
    function setDeviceInfo() {
      const icons = {
        windows: '<i class="ri-microsoft-fill"></i>',
        android: '<i class="ri-android-fill"></i>',
        apple: '<i class="ri-apple-fill"></i>'
      };
      if (navigator.userAgent.match(/Android/i)) {
        return `${icons.android} Instalar en Android`;
      }
      if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {
        return `${icons.apple} Instalar en iOS`;
      }
      return `${icons.windows} Instalar en Windows`;
    }

    // Función para instalar PWA en el navegador
    function installPWA() {
      const container = document.getElementById('install-container');
      const close = document.getElementById('close-install');
      const install = document.getElementById('install-pwa');
      const checkIosDevice = () => {
        const ua = navigator.userAgent;
        const vendor = navigator.vendor;
        const isSafari = /Safari/.test(ua) && /Apple Computer/.test(vendor) && !/CriOS/.test(ua); // Excluir Chrome

        return isSafari;
      };
      if (checkIosDevice()) return;
      if (!container) return;
      if (!install) return;
      close.addEventListener('click', () => {
        container.classList.remove('is-active');
      });
      install.innerHTML = setDeviceInfo();
      const isMobileDevice = () => {
        const mobileDeviceRegex = /Android|Windows Phone|Windows|iPhone|iPad|iPod|iOS|Macintosh/i;
        return mobileDeviceRegex.test(navigator.userAgent);
      };
      const isFirefox = () => {
        return navigator.userAgent.includes('Firefox');
      };
      let installPrompt = null;
      window.addEventListener('beforeinstallprompt', event => {
        event.preventDefault();
        installPrompt = event;
        if (isMobileDevice() && !isFirefox()) {
          container.classList.add('is-active');
        }
      });
      install.addEventListener('click', async () => {
        if (!installPrompt) {
          console.warn('No se puede instalar la PWA');
          return;
        }
        const result = await installPrompt.prompt();
        if (result.outcome === 'accepted') {
          console.warn('PWA instalada');
          container.classList.remove('is-active');
        } else {
          console.warn('PWA no instalada');
        }
      });
    }
    installPWA();
    function convertSecondsToMinutes(seconds) {
      const minutes = Math.floor(seconds / 60);
      const remainingSeconds = seconds % 60;
      return `${minutes}:${remainingSeconds.toFixed(0) < 10 ? '0' : ''}${remainingSeconds.toFixed(0)}`;
    }

    // Obtener porcentaje entre dos valores
    function getPercentage(value, total) {
      return value * 100 / total;
    }
    function songProgress(duration, elapsed) {
      const progress = getPercentage(elapsed, duration);

      // Agregar la variable css con el progreso de la canción
      player.style.setProperty('--song-progress', `${2 + progress}%`);

      // Actualizar el progreso de la canción
      const songElapsed = document.querySelector('.song-elapsed');
      const songDuration = document.querySelector('.song-duration');
      if (songElapsed) {
        songElapsed.textContent = convertSecondsToMinutes(elapsed);
      }
      if (songDuration) {
        songDuration.textContent = convertSecondsToMinutes(duration);
      }
    }

    // Obtener datos de la estación
    async function initApp() {
      const response = await fetch(dataUrl);
      const data = await response.json();
      if (isMultiStream) {
        const stations = data.map(station => getStationData(station, baseUrl, true));
        loadStations = stations;
        setupAll(loadStations[0]);
      } else {
        const station = getStationData(data, baseUrl);
        currentStation = station;
        setupAll(station);
      }

      // Iniciar el stream
      async function init(current) {
        // Incrementar el token de actualización
        const token = ++currentUpdateToken;
        currentStation = current;

        // Cancelar el timeout anterior
        if (timeoutId) clearTimeout(timeoutId);
        const song = await getStreamData(current.stream_url);

        // Si en el intento se cambió la estación, se descarta esta respuesta
        if (token !== currentUpdateToken) return;
        if (song.now_playing) {
          songProgress(song.now_playing.duration, song.now_playing.elapsed);
        }
        const title = song.title || song.song;
        const query = title.trim() !== '' ? `${song.artist} - ${title}` : `${current.name} - ${current.descripcion}`;
        if (currentSongPlaying !== query) {
          currentSongPlaying = query;
          const songData = await getDataFromSearchApi(query, SERVICE);
          if (songData.artwork.includes('https://radioapi.lat/') || songData.artwork === null || title.trim() === '') {
            songData.artwork = current.picture || current.cover;
            songData.title = current.name;
            songData.artist = current.description;
          } else {
            songData.title = title;
            songData.artist = song.artist;
          }
          setCurrentSong(songData, currentStation.picture, token);
          setMediaSession(songData, currentStation.picture, token);
          setHistory(song.song_history, currentStation, token);
          const buttonShare = document.querySelector('.player-button-share');
          if (buttonShare) {
            buttonShare.onclick = () => {
              openModal('share', {
                data: songData
              });
            };
          }
          const lyricsData = await getLyrics(song.artist, title);
          const lyricsButton = document.querySelector('.player-button-lyrics');
          if (lyricsButton) {
            lyricsButton.onclick = () => {
              openModal('lyrics', {
                lyrics: lyricsData
              });
            };
          }
        }
        timeoutId = setTimeout(() => {
          init(current);
        }, TIMEOUT);
      }

      // DATOS DE LA PROGRAMACIÓN
      function initProgram() {
        if (timeoutIdProgram) clearTimeout(timeoutIdProgram);
        fetch(dataProgramUrl).then(response => response.json()).then(data => {
          setupSchedule(data, baseUrl);
        }).catch(error => {
          console.error('Error:', error);
        });
        // Refrescar cada 1 minuto
        timeoutIdProgram = setTimeout(() => {
          initProgram();
        }, 60000);
      }
      initProgram();

      // DATOS DE LOS BANNERS
      fetch(dataBannersUrl).then(response => response.json()).then(data => {
        const container = document.querySelector('.slider-wrapper');
        setupSlider(container, data, baseUrl);
      }).catch(error => {
        console.error('Error:', error);
      });

      // Botones sociales
      fetch(dataSocialUrl).then(response => response.json()).then(data => {
        setupSocial(playerSocial, getSocials(data));
      }).catch(error => {
        console.error('Error:', error);
      });
      const promoContainer = document.getElementById('player-prm-send');
      if (promoContainer) {
        // DATOS DE LAS PROMOCIONES
        fetch(dataPromoUrl).then(response => response.json()).then(data => {
          if (data.length === 0) return;
          const button = document.querySelector('.send-prm');
          button.classList.remove('none');
          data.forEach(item => {
            const img = document.createElement('img');
            img.src = baseUrl + item.photo;
            img.classList.add('prm-image');
            img.addEventListener('click', () => {
              openModal('form', {
                formAction: dataGetPromoUrl,
                formId: item.promocion_id
              });
            });
            promoContainer.appendChild(img);
          });
        }).catch(error => {
          console.error('Error:', error);
        });
      }

      // DATOS DE LAS NOTICIAS
      function initNoticias() {
        const newsContainer = document.querySelector('.main-news');
        if (!newsContainer) return;
        fetch(dataNoticiasUrl).then(response => response.json()).then(data => {
          if (data.length === 0) return;
          setArticles(data, baseUrl);
          newsContainer.classList.remove('none');
        }).catch(error => {
          console.error('Error:', error);
        });
      }
      if (MODULE_NEWS) initNoticias();

      // DATOS DE LAS NOTICIAS
      function initTeam() {
        const teamContainer = document.querySelector('.main-team');
        if (!teamContainer) return;
        fetch(dataTeamUrl).then(response => response.json()).then(data => {
          if (data.length === 0) return;
          setTeam(data, baseUrl);
          teamContainer.classList.remove('none');
        }).catch(error => {
          console.error('Error:', error);
        });
      }
      if (MODULE_TEAM) initTeam();
      function initVideoApp() {
        const container = document.querySelector('.main-ytlist');
        if (!container) return;
        const CONTAINER_LIST = container.querySelector('.ytlistmini-list');
        if (!CONTAINER_LIST) return;
        fetch(dataVideoUrl).then(response => response.json()).then(data => {
          container.classList.remove('none');
          fetchVideoList(data, CONTAINER_LIST);
        }).catch(error => {
          console.error('Error:', error);
        });
      }
      if (MODULE_VIDEO_TOPS) initVideoApp();

      // Estaciones
      if (loadStations.length > 1) {
        stationsButton.classList.remove('none');
        loadStations.forEach((station, index) => {
          const img = document.createElement('img');
          img.classList.add('prm-image');
          img.src = station.picture;
          if (index === 0) {
            img.classList.add('is-current');
            currentActiveStation = img;
          }
          img.addEventListener('click', event => {
            if (currentStation === station) return;
            if (currentActiveStation) {
              currentActiveStation.classList.remove('is-current');
            }
            currentActiveStation = img;
            currentActiveStation.classList.add('is-current');
            init(station);
            setupAll(station);
            play(playButton, station.stream_url);
            currentStation = station;
          });
          stationsContainer.appendChild(img);
        });
      }

      // Iniciar todo
      if (isMultiStream) {
        init(loadStations[0]);
      } else {
        init(currentStation);
      }
    }
    initApp();
  }

  function playerstatic() {
    const buttons = document.querySelectorAll('[data-outside]');
    const ACTIVE_CLASS = 'is-active';
    const SERVICE = window?.streams?.service || 'spotify';

    // Delete the class 'preload' from the body when the page is loaded
    window.addEventListener('DOMContentLoaded', event => {
      document.body.classList.remove('preload');
    });

    // Agrega o quita la clase ACTIVE_CLASS a los elementos
    function outsideClick(button) {
      if (!button) return;
      const target = document.getElementById(button.dataset.outside);
      if (!target) return;
      button.addEventListener('click', () => {
        button.classList.toggle(ACTIVE_CLASS);
        target.classList.toggle(ACTIVE_CLASS);
      });
      const clickOutside = event => {
        if (!target.contains(event.target) && !button.contains(event.target) && !modal.contains(event.target)) {
          button.classList.remove(ACTIVE_CLASS);
          target.classList.remove(ACTIVE_CLASS);
        }
      };
      document.addEventListener('click', clickOutside);
      const close = target.querySelector('[data-close]');
      if (close) {
        close.onclick = function () {
          button.classList.remove(ACTIVE_CLASS);
          target.classList.remove(ACTIVE_CLASS);
        };
      }
    }

    // Recorrer todos los botones que tengan el atributo data-outside
    buttons.forEach(button => {
      outsideClick(button);
    });

    // Función para inicializar el lienzo (canvas)
    function initCanvas(container) {
      const canvas = document.createElement('canvas');
      canvas.setAttribute('id', 'visualizerCanvas');
      canvas.setAttribute('class', 'visualizer-item');
      container.appendChild(canvas);
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
      return canvas;
    }

    // Función para cambiar el lienzo según el tamaño del contenedor
    function resizeCanvas(canvas, container) {
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
    }

    // Visualizer
    const visualizer = (audio, container) => {
      if (!audio || !container) {
        return;
      }
      const options = {
        fftSize: container.dataset.fftSize || 2048,
        numBars: container.dataset.bars || 80,
        maxHeight: container.dataset.maxHeight || 255
      };
      const ctx = new AudioContext();
      const audioSource = ctx.createMediaElementSource(audio);
      const analyzer = ctx.createAnalyser();
      audioSource.connect(analyzer);
      audioSource.connect(ctx.destination);
      const frequencyData = new Uint8Array(analyzer.frequencyBinCount);
      const canvas = initCanvas(container);
      const canvasCtx = canvas.getContext('2d');

      // Crear barras
      const renderBars = () => {
        resizeCanvas(canvas, container);
        analyzer.getByteFrequencyData(frequencyData);
        {
          analyzer.fftSize = options.fftSize;
        }
        canvasCtx.clearRect(0, 0, canvas.width, canvas.height);
        for (let i = 0; i < options.numBars; i++) {
          const index = Math.floor((i + 10) * (i < options.numBars / 2 ? 2 : 1));
          const fd = frequencyData[index];
          const barHeight = Math.max(4, fd || 0) + options.maxHeight / 255;
          const barWidth = canvas.width / options.numBars;
          const x = i * barWidth;
          const y = canvas.height - barHeight;
          canvasCtx.fillStyle = 'white';
          canvasCtx.fillRect(x, y, barWidth - 2, barHeight);
        }
        requestAnimationFrame(renderBars);
      };
      renderBars();

      // Listener del cambio de espacio en la ventana
      window.addEventListener('resize', () => {
        resizeCanvas(canvas, container);
      });
    };
    const API_KEY_LYRICS = '1637b78dc3b129e6843ed674489a92d0';
    const cache = {};

    // Iconos de Meteor Icons: https://meteoricons.com/
    const icons = {
      play: '<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M9.8125 9.315v5.9025c0 .2925.1575.56.4125.7.255.14.5625.1375.815-.0175l4.83-2.9525c.2375-.1475.385-.405.385-.6875s-.1475-.54-.385-.6875L11.04 8.62c-.2475-.15-.56-.1575-.815-.0175V8.61a.8023.8023 90 00-.4125.7M0 12a.25.25 90 0124 0A.25.25 90 010 12Z"></path></svg>',
      pause: '<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path stroke="none" d="M0 12a.25.25 90 0124 0A.25.25 90 010 12Zm9 2a1 1 0 001 1h4a1 1 0 001-1V10A1 1 0 0014 9H10A1 1 0 009 10Z"></path></svg>',
      facebook: '<svg class="i i-facebook" viewBox="0 0 24 24"><path d="M17 14h-3v8h-4v-8H7v-4h3V7a5 5 0 0 1 5-5h3v4h-3q-1 0-1 1v3h4Z"></path></svg>',
      twitter: '<svg class="i i-x" viewBox="0 0 24 24"><path d="m3 21 7.5-7.5m3-3L21 3M8 3H3l13 18h5Z"></path></svg>',
      instagram: '<svg class="i i-instagram" viewBox="0 0 24 24"><circle cx="12" cy="12" r="4"></circle><rect width="20" height="20" x="2" y="2" rx="5"></rect><path d="M17.5 6.5h0"></path></svg>',
      youtube: '<svg class="i i-youtube" viewBox="0 0 24 24"><path d="M1.5 17q-1-5.5 0-10Q1.9 4.8 4 4.5q8-1 16 0 2.1.3 2.5 2.5 1 4.5 0 10-.4 2.2-2.5 2.5-8 1-16 0-2.1-.3-2.5-2.5Zm8-8.5v7l6-3.5Z"></path></svg>',
      tiktok: '<svg class="i i-tiktok" viewBox="0 0 24 24"><path d="M22 6v5q-4 0-6-2v7a7 7 0 1 1-5-6.7m0 6.7a2 2 0 1 0-2 2 2 2 0 0 0 2-2V1h5q2 5 6 5"></path></svg>',
      whatsapp: '<svg class="i i-whatsapp" viewBox="0 0 24 24"><circle cx="9" cy="9" r="1"></circle><circle cx="15" cy="15" r="1"></circle><path d="M8 9a7 7 0 0 0 7 7m-9 5.2A11 11 0 1 0 2.8 18L2 22Z"></path></svg>',
      telegram: '<svg class="i i-telegram" viewBox="0 0 24 24"><path d="M12.5 16 9 19.5 7 13l-5.5-2 21-8-4 18-7.5-7 4-3"></path></svg>',
      tv: '<svg class="i i-tv" viewBox="0 0 24 24"><title>video</title><path d="M17,10.5V7A1,1 0 0,0 16,6H4A1,1 0 0,0 3,7V17A1,1 0 0,0 4,18H16A1,1 0 0,0 17,17V13.5L21,17.5V6.5L17,10.5Z" /></svg>',
      windows: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M11.501 3V11.5H3.00098V3H11.501ZM11.501 21H3.00098V12.5H11.501V21ZM12.501 3H21.001V11.5H12.501V3ZM21.001 12.5V21H12.501V12.5H21.001Z"></path></svg>',
      apple: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M11.6734 7.22198C10.7974 7.22198 9.44138 6.22598 8.01338 6.26198C6.12938 6.28598 4.40138 7.35397 3.42938 9.04597C1.47338 12.442 2.92538 17.458 4.83338 20.218C5.76938 21.562 6.87338 23.074 8.33738 23.026C9.74138 22.966 10.2694 22.114 11.9734 22.114C13.6654 22.114 14.1454 23.026 15.6334 22.99C17.1454 22.966 18.1054 21.622 19.0294 20.266C20.0974 18.706 20.5414 17.194 20.5654 17.11C20.5294 17.098 17.6254 15.982 17.5894 12.622C17.5654 9.81397 19.8814 8.46998 19.9894 8.40998C18.6694 6.47798 16.6414 6.26198 15.9334 6.21398C14.0854 6.06998 12.5374 7.22198 11.6734 7.22198ZM14.7934 4.38998C15.5734 3.45398 16.0894 2.14598 15.9454 0.849976C14.8294 0.897976 13.4854 1.59398 12.6814 2.52998C11.9614 3.35798 11.3374 4.68998 11.5054 5.96198C12.7414 6.05798 14.0134 5.32598 14.7934 4.38998Z"></path></svg>',
      android: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M6.38231 3.9681C7.92199 2.73647 9.87499 2 12 2C14.125 2 16.078 2.73647 17.6177 3.9681L19.0711 2.51472L20.4853 3.92893L19.0319 5.38231C20.2635 6.92199 21 8.87499 21 11V12H3V11C3 8.87499 3.73647 6.92199 4.9681 5.38231L3.51472 3.92893L4.92893 2.51472L6.38231 3.9681ZM3 14H21V21C21 21.5523 20.5523 22 20 22H4C3.44772 22 3 21.5523 3 21V14ZM9 9C9.55228 9 10 8.55228 10 8C10 7.44772 9.55228 7 9 7C8.44772 7 8 7.44772 8 8C8 8.55228 8.44772 9 9 9ZM15 9C15.5523 9 16 8.55228 16 8C16 7.44772 15.5523 7 15 7C14.4477 7 14 7.44772 14 8C14 8.55228 14.4477 9 15 9Z"></path></svg>'
    };
    const svgIcons = {
      deezer: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" d="M20.1 3.6c.2-1.2.6-2 1-2 .6 0 1.1 2.8 1.1 6.2 0 3.5-.5 6.3-1.2 6.3-.3 0-.5-.5-.7-1.3-.3 3-1 5-1.8 5-.6 0-1.1-1.3-1.5-3.2-.3 3.7-.9 6.2-1.6 6.2-.5 0-.9-1-1.2-2.7C13.8 21.6 13 24 12 24c-1 0-1.9-2.3-2.3-5.8-.3 1.7-.7 2.7-1.1 2.7-.8 0-1.4-2.6-1.6-6.2-.4 2-1 3.1-1.5 3.1-.8 0-1.5-2-1.8-4.9-.2.8-.5 1.3-.8 1.3-.6 0-1.2-2.8-1.2-6.3 0-3.4.6-6.2 1.2-6.2.4 0 .7.8 1 2C4.1 1.5 4.8 0 5.4 0c.7 0 1.4 2 1.7 5 .4-2.2.8-3.5 1.4-3.5.7 0 1.4 2.7 1.6 6.4.4-1.9 1-3.1 1.8-3.1.7 0 1.4 1.2 1.8 3.1.2-3.7.9-6.4 1.6-6.4.5 0 1 1.3 1.3 3.5.3-3 1-5 1.8-5 .7 0 1.2 1.4 1.6 3.6zM.7 10C.3 10 0 8.7 0 7.2c0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8S1.1 10 .7 10zm22.6 0c-.4 0-.7-1.3-.7-2.8 0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8s-.3 2.8-.7 2.8z"/></svg>'
    };
    const getDataFromSearchApi = async (query, service) => {
      if (cache[query]) {
        return cache[query];
      }
      const streamUrl = `api/search.php?query=${encodeURIComponent(query)}&service=${service}`;
      const response = await fetch(streamUrl);
      const data = await response.json();

      // Si no responde
      if (!data.results) {
        return {};
      }
      const results = data.results;
      cache[query] = results;
      return results;
    };

    // Obtener letras de canciones
    const getLyrics = async (artist, name) => {
      try {
        const response = await fetch(`https://api.vagalume.com.br/search.php?apikey=${API_KEY_LYRICS}&art=${encodeURIComponent(artist)}&mus=${encodeURIComponent(name)}`);
        const data = await response.json();
        if (data.type === 'exact' || data.type === 'aprox') {
          const lyrics = data.mus[0].text;
          return lyrics;
        } else {
          return 'Letra no disponible';
        }
      } catch (error) {
        console.error('Error fetching lyrics:', error);
        return 'Letra no disponible';
      }
    };

    // Eliminar elementos innecesarios del texto
    function sanitizeText(text) {
      return text.replace(/^\d+\.\)\s/, '').replace(/<br>$/, '');
    }

    // Normalizar historial
    function normalizeHistory(api) {
      let artist;
      let song;
      let history = api.song_history || api.history || api.songHistory || [];
      history = history.slice(0, 10);
      const historyNormalized = history.map(item => {
        if (api.song_history) {
          artist = item.song.artist;
          song = item.song.title;
        } else if (api.history) {
          artist = sanitizeText(item.split(' - ')[0] || item);
          song = sanitizeText(item.split(' - ')[1] || item);
        } else if (api.songHistory) {
          artist = item.artist;
          song = item.title;
        }
        return {
          artist,
          song
        };
      });

      // limitar a 4 elementos
      return historyNormalized;
    }
    const playButton = document.querySelector('.player-button-play');
    const visualizerContainer = document.querySelector('.visualizer');
    const audio = new Audio();
    audio.crossOrigin = 'anonymous';
    let hasVisualizer = false;
    function play(audio, newSource = null) {
      if (newSource) {
        audio.src = newSource;
      }

      // Visualizer
      if (!hasVisualizer) {
        visualizer(audio, visualizerContainer);
        hasVisualizer = true;
      }
      audio.load();
      audio.play();
      playButton.innerHTML = icons.pause;
      playButton.classList.add('is-active');
      document.body.classList.add('is-playing');
    }
    function pause(audio) {
      audio.pause();
      playButton.innerHTML = icons.play;
      playButton.classList.remove('is-active');
      document.body.classList.remove('is-playing');
    }

    // Botón play/pause, al pausar detener el stream, al reproducir iniciar el stream de nuevo
    // playButton, play, pause son funciones exportadas que se usaran en otros archivos

    if (playButton !== null) {
      playButton.addEventListener('click', async () => {
        if (audio.paused) {
          play(audio);
        } else {
          pause(audio);
        }
      });
    }

    // Botón play/pause, al pausar detener el stream, al reproducir iniciar el stream de nuevo
    // playButton, play, pause son funciones exportadas que se usaran en otros archivos
    const range = document.querySelector('.player-volume');
    const rangeFill = document.querySelector('.player-range-fill');
    const rangeWrapper = document.querySelector('.player-range-wrapper');
    const rangeThumb = document.querySelector('.player-range-thumb');
    let currentVolume;
    try {
      currentVolume = localStorage.getItem('volume') || 100;
    } catch (error) {
      currentVolume = 100;
      console.warn('Error al obtener el volumen del localStorage:', error);
    }

    // Rango recorrido
    function setRangeWidth(percent) {
      {
        rangeFill.style.height = `${percent}%`;
      }
    }

    // Posición del thumb
    function setThumbPosition(percent) {
      const compensatedWidth = rangeWrapper.offsetHeight - rangeThumb.offsetHeight ;
      const thumbPosition = percent / 100 * compensatedWidth;
      {
        rangeThumb.style.bottom = `${thumbPosition}px`;
      }
    }

    // Actualiza el volumen al cambiar el rango
    function updateVolume(audio, value) {
      range.value = value;
      setRangeWidth(value);
      setThumbPosition(value);
      audio.volume = value / 100;
      if (localStorage === null || localStorage === undefined) return;
      localStorage.setItem('volume', value);
    }

    // Valor inicial
    if (range !== null) {
      updateVolume(audio, currentVolume);

      // Escucha el cambio del rango
      range.addEventListener('input', event => {
        updateVolume(audio, event.target.value);
      });

      // Escucha el movimiento del mouse
      rangeThumb.addEventListener('mousedown', () => {
        document.addEventListener('mousemove', handleThumbDrag);
      });
    }

    // Mueve el thumb y actualiza el volumen
    function handleThumbDrag(event) {
      const rangeRect = range.getBoundingClientRect();
      const click = event.clientY - rangeRect.top;
      let percent = click / range.offsetWidth * 100;
      percent = 100 - percent;
      percent = Math.max(0, Math.min(100, percent));
      const value = Math.round((range.max - range.min) * (percent / 100)) + parseInt(range.min);
      updateVolume(audio, value);
    }

    // Deja de escuchar el movimiento del mouse
    document.addEventListener('mouseup', () => {
      document.removeEventListener('mousemove', handleThumbDrag);
    });
    window.addEventListener('resize', () => {
      const currentPercent = range.value;
      setRangeWidth(currentPercent);
      setThumbPosition(currentPercent);
    });
    const songNow = document.querySelector('.song-now');
    const stationsList = document.getElementById('stations');
    const stationName = document.querySelector('.station-name');
    const stationDescription = document.querySelector('.station-description');
    const playerCoverImg = document.querySelector('.player-cover-image');
    const playerSocial = document.querySelector('.player-social');
    const playerTv = document.querySelector('.footer-tv');
    const playerTvModal = document.getElementById('modal-tv');
    const playerProgram = document.querySelector('.player-program');
    const lyricsContent = document.getElementById('lyrics');
    const history = document.getElementById('history');
    const API_URL = 'api/met?url=';
    const TIME_TO_REFRESH = window?.streams?.timeRefresh || 10000;
    let currentStation;
    let activeButton;
    function setAssetsInPage(station) {
      const playerArtwork = document.querySelector('.player-artwork img:first-of-type');
      playerTv.innerHTML = '';
      playerArtwork.src = station.cover || station.picture;
      playerCoverImg.src = station.cover || station.picture;
      stationName.textContent = station.name;
      stationDescription.textContent = station.description;
      if (station.tv_url && playerTv) {
        createOpenTvButton(station.tv_url);
      }
    }

    // Crear una imagen proxy
    // @param {string} src - URL de la imagen
    // @returns {Promise} - Promesa que se resuelve con la imagen
    function createProxyImage(src) {
      return new Promise((resolve, reject) => {
        const img = document.createElement('img');
        img.crossOrigin = 'Anonymous';
        img.src = `https://images.weserv.nl/?url=${src}`;
        img.onload = () => resolve(img);
        img.onerror = reject;
      });
    }

    // Establecer el color de acento de la página
    // @param {HTMLImageElement} image - Imagen a extraer el color
    // @param {ColorThief} colorThief - Instancia de ColorThief
    function setAccentColor(image, colorThief) {
      const metaThemeColor = document.querySelector('meta[name=theme-color]');

      // Función para establecer el color
      const applyColor = () => {
        const color = `rgb(${colorThief.getColor(image)})`;
        document.querySelector('html').style.setProperty('--accent', color);
        if (metaThemeColor) {
          metaThemeColor.setAttribute('content', color);
        }
      };
      if (image.complete) {
        applyColor();
      } else {
        image.addEventListener('load', applyColor);
      }
    }
    const modal = document.querySelector('.modal');
    const modalBody = modal.querySelector('.modal-body');
    const OPEN_CLASS = 'is-open';
    function openSongModal({
      artist,
      title,
      artwork,
      genre,
      album
    }) {
      modalBody.innerHTML = `<div class="modal-song-info p-1 text-center">
      <img src="${artwork}" alt="portada modal" class="player-modal-image" width="200" height="200">
      <h3 class="fs-4 fw-700 color-text-inverse song-name">${title}</h3>
      <div class="flex column color-text-inverse justify-center modal-social player-socials wrap">
        <span class="fs-5"><strong class="color-text-inverse fw-700">Artista: </strong>${artist}</span>
        <span class="fs-5"><strong class="color-text-inverse fw-700">Genero: </strong>${genre}</span>
        <span class="fs-5"><strong class="color-text-inverse fw-700">Album: </strong>${album}</span>
      </div>
    </div>
    `;
    }
    function openModal(type, options = {}) {
      if (!modal) return;
      modal.classList.add(OPEN_CLASS);
      modalBody.innerHTML = '';
      modal.classList.add(`modal-${type}`);
      switch (type) {
        case 'song':
          openSongModal(options.data);
          break;
        default:
          console.warn('Tipo de modal desconocido:', type);
          return;
      }
      function closeModal() {
        modal.classList.remove(OPEN_CLASS, `modal-${type}`);
        if (currentVideo) currentVideo.dispose();
        modalBody.innerHTML = '';
        if (options.hook && typeof options.hook === 'function') {
          options.hook();
        }
      }

      // Cerrar
      const closeBtn = modal.querySelector('.modal-close');
      if (closeBtn) {
        closeBtn.addEventListener('click', closeModal);
      }
      modal.addEventListener('click', e => {
        if (!modal.querySelector('.modal-content').contains(e.target)) closeModal();
      });
    }
    let currentVideo;
    function initVideoPlayer(videoUrl, container) {
      const $video = document.createElement('video');
      $video.id = 'player';
      $video.classList.add('video-js', 'video-js-stream');
      container.innerHTML = '';
      container.appendChild($video);
      const videoId = document.getElementById('player');

      // eslint-disable-next-line no-undef
      const video = videojs(videoId, {
        sources: [{
          src: videoUrl,
          type: 'application/x-mpegURL'
        }],
        controls: true,
        autoplay: true
      });
      currentVideo = video;
    }
    function createOpenTvButton(url) {
      const $button = document.createElement('button');
      $button.classList.add('player-button-tv', 'btn');
      $button.innerHTML = icons.tv + 'TV EN VIVO';
      const modalBody = playerTvModal.querySelector('.modal-body-video');
      $button.addEventListener('click', () => {
        playerTvModal.classList.add('is-active');
        pause(audio);
        initVideoPlayer(url, modalBody);
      });
      const $closeButton = playerTvModal.querySelector('[data-close]');
      $closeButton.addEventListener('click', () => {
        if (currentVideo) {
          currentVideo.dispose();
        }
        modalBody.innerHTML = '';
        playerTvModal.classList.remove('is-active');
        play(audio);
      });
      playerTv.appendChild($button);
    }
    function createProgram(program) {
      if (!program) return;
      playerProgram.innerHTML = '';
      const $div = document.createElement('div');
      const $span = document.createElement('span');
      $div.classList.add('player-program-time-container');
      $span.classList.add('player-program-badge');
      $span.textContent = 'On Air';
      $div.appendChild($span);
      const $time = document.createElement('span');
      $time.classList.add('player-program-time', 'fs-8');
      $time.textContent = "REPRODUCIENDO";
      $div.appendChild($time);
      playerProgram.appendChild($div);
      if (program.name) {
        const $name = document.createElement('span');
        $name.classList.add('player-program-name');
        $name.textContent = program.name;
        playerProgram.appendChild($name);
      }
      if (program.description) {
        const $description = document.createElement('span');
        $description.classList.add('player-program-description');
        $description.textContent = program.description;
        playerProgram.appendChild($description);
      }
    }
    function createStations(stations, currentStation, callback) {
      if (!stationsList) return;
      stationsList.innerHTML = '';
      stations.forEach(async (station, index) => {
        const $fragment = document.createDocumentFragment();
        const $button = createStreamItem(station, index, currentStation, callback);
        $fragment.appendChild($button);
        stationsList.appendChild($fragment);
      });
    }
    function createStreamItem(station, index, currentStation, callback) {
      const $button = document.createElement('button');
      $button.classList.add('station');
      $button.innerHTML = `<img class="station-img" src="${station.cover}" alt="station" height="160" width="160">`;
      $button.dataset.index = index;
      $button.dataset.hash = station.hash;
      if (currentStation.stream_url === station.stream_url) {
        $button.classList.add('is-active');
        activeButton = $button;
      }
      $button.addEventListener('click', () => {
        if ($button.classList.contains('is-active')) return;

        // Eliminar la clase "active" del botón activo anterior, si existe
        if (activeButton) {
          activeButton.classList.remove('is-active');
        }

        // Agregar la clase "active" al botón actualmente presionado
        $button.classList.add('is-active');
        activeButton = $button; // Actualizar el botón activo

        setAssetsInPage(station);
        play(audio, station.stream_url);
        if (history) {
          history.innerHTML = '';
        }

        // Llamar a la función de devolución de llamada (callback) si se proporciona
        if (typeof callback === 'function') {
          callback(station);
        }
      });
      return $button;
    }

    // Cargar datos de la canción actual al navegador
    function setMediaSession(data, picture) {
      const {
        title,
        artist,
        artwork
      } = data;
      if ('mediaSession' in navigator) {
        navigator.mediaSession.metadata = new MediaMetadata({
          title,
          artist,
          artwork: [{
            src: artwork || picture,
            sizes: '128x128',
            type: 'image/png'
          }, {
            src: artwork || picture,
            sizes: '256x256',
            type: 'image/png'
          }, {
            src: artwork || picture,
            sizes: '512x512',
            type: 'image/png'
          }]
        });
        navigator.mediaSession.setActionHandler('play', () => {
          play();
        });
        navigator.mediaSession.setActionHandler('pause', () => {
          pause();
        });
      }
    }
    function scrollText(container) {
      const item = container.querySelector('.song-name');
      if (item.scrollWidth > container.offsetWidth) {
        item.classList.add('is-scrolling');
        const scroll = item.scrollWidth - container.offsetWidth;
        const duration = 5000 + scroll * 100;
        item.setAttribute('style', `--text-scroll-duration: ${duration}ms; --text-scroll: -${scroll}px`);
        return;
      }
      item.classList.remove('is-scrolling');
      item.removeAttribute('style');
    }
    // Obtener porcentaje entre dos valores
    function getPercentage(value, total) {
      return value * 100 / total;
    }
    function convertSecondsToMinutes(seconds) {
      const minutes = Math.floor(seconds / 60);
      const remainingSeconds = seconds % 60;
      return `${minutes}:${remainingSeconds.toFixed(0) < 10 ? '0' : ''}${remainingSeconds.toFixed(0)}`;
    }
    function songProgress(duration, elapsed) {
      const progress = getPercentage(elapsed, duration);
      const appSuper = document.querySelector('.app');

      // Agregar la variable css con el progreso de la canción
      appSuper.style.setProperty('--song-progress', `${2 + progress}%`);

      // Actualizar el progreso de la canción
      const songElapsed = document.querySelector('.song-elapsed');
      const songDuration = document.querySelector('.song-duration');
      if (songElapsed) {
        songElapsed.textContent = convertSecondsToMinutes(elapsed);
      }
      if (songDuration) {
        songDuration.textContent = convertSecondsToMinutes(duration);
      }
    }
    function loadImage(src) {
      return new Promise((resolve, reject) => {
        const img = new Image();
        img.onload = () => resolve(img);
        img.onerror = reject;
        img.src = src;
      });
    }
    function slideUpImageTransition(container, src) {
      const img = document.createElement('img');
      const size = container.clientHeight;
      img.src = src;
      img.width = size;
      img.height = size;
      container.appendChild(img);
      const firstImg = container.querySelector('img:first-of-type');
      firstImg.style.marginLeft = `-${size}px`;
      firstImg.addEventListener('transitionend', () => {
        const allImgExcLast = container.querySelectorAll('img:not(:last-child)');
        allImgExcLast.forEach(img => img.remove());
      }, {
        once: true
      });
    }

    // Establecer datos de la canción actual
    function currentSong(data, picture) {
      const content = songNow;
      content.querySelector('.song-name').textContent = data.title || data.song;
      content.querySelector('.song-artist').textContent = data.artist;
      const parentTitle = content.querySelector('.player-cover-title');
      scrollText(parentTitle);
      const pictureEl = document.querySelector('.player-artwork');
      const artworkUrl = data.artwork || picture;

      // eslint-disable-next-line no-undef
      const colorThief = new ColorThief();

      // Establecer el color de acento
      createProxyImage(artworkUrl).then(img => {
        setAccentColor(img, colorThief);
      }).catch(() => {
        console.error('Error al cargar la imagen');
      });
      if (pictureEl) {
        loadImage(artworkUrl).then(() => {
          slideUpImageTransition(pictureEl, artworkUrl);
        }).catch(() => {
          console.error('Error al cargar arte de la canción');
        });
      }
      if (playerCoverImg) {
        const tempImg = new Image();
        tempImg.src = artworkUrl;
        tempImg.addEventListener('load', () => {
          playerCoverImg.style.opacity = 0;

          // Esperar a que la animación termine
          playerCoverImg.addEventListener('transitionend', () => {
            playerCoverImg.src = artworkUrl;
            playerCoverImg.style.opacity = 1;
          });
        });
      }
    }
    function getSocials(data) {
      const socials = {};
      if (data.radio_facebook_url) {
        socials.facebook = data.radio_facebook_url;
      }
      if (data.radio_youtube) {
        socials.youtube = data.radio_youtube;
      }
      if (data.radio_instagram) {
        socials.instagram = data.radio_instagram;
      }
      if (data.radio_whatsapp) {
        socials.whatsapp = `https://wa.me/+${data.radio_whatsapp}`;
      }
      if (data.radio_menu_noticias) {
        socials.tiktok = data.radio_menu_noticias;
      }
      return socials;
    }
    function getApps(data) {
      const apps = {};
      // Verifica si existe el campo para la app Android
      if (data.radio_fondo_color) {
        apps.android = data.radio_fondo_color; // Asigna la URL para la app Android
      }
      // Verifica si existe el campo para la app iOS
      if (data.radio_splash_color) {
        apps.ios = data.radio_splash_color; // Asigna la URL para la app iOS
      }
      return apps;
    }

    // Establecer las canciones que se han reproducido
    async function setHistory(data, current) {
      const historyContainer = document.querySelector('.last-played');
      if (!historyContainer || !data) return;

      // Array para almacenar los datos de las canciones
      const historyCurrentData = [];
      const itemsHTML = await Promise.all(data.map(async ({
        song: {
          artist,
          title
        }
      }, index) => {
        const dataFrom = await getDataFromSearchApi(`${artist} - ${title}`, SERVICE);
        if (dataFrom.artwork.includes('https://nexoapi.lat/') || dataFrom.artwork === null || title.trim() === '') {
          dataFrom.artwork = current.picture || current.cover;
          dataFrom.title = current.name;
          dataFrom.artist = current.description;
        } else {
          dataFrom.title = title;
          dataFrom.artist = artist;
        }

        // Almacenar los datos en el índice correcto
        historyCurrentData[index] = dataFrom;
        return `
          <div class="history" data-index="${index}">
            <div class="history-wrapper">
              <img class="history-art" src="${dataFrom.artwork}" alt="${dataFrom.title}">
              <div class="history-meta fs-7">
                <span class="history-title fw-700 uppercase">${dataFrom.title}</span>
                <span class="history-artist">${dataFrom.artist}</span>
              </div>
            </div>
            <a class="history-url history-deezer" href="${dataFrom.stream || '#not-found'}" target="_blank">
              ${svgIcons.deezer}
            </a>
            <button class="history-button"></button>
          </div>`;
      }));
      historyContainer.innerHTML = itemsHTML.join('');
      const historyItemsElements = historyContainer.querySelectorAll('.history');
      historyItemsElements.forEach(item => {
        const button = item.querySelector('.history-button');
        const index = item.getAttribute('data-index');
        button.addEventListener('click', event => {
          // Abrir modal con la canción seleccionada
          event.preventDefault();
          openModal('song', {
            data: historyCurrentData[index]
          });
        });
      });
    }
    function setLyrics(artist, title) {
      if (!lyricsContent) return;
      getLyrics(artist, title).then(lyrics => {
        const $p = document.createElement('p');
        $p.innerHTML = lyrics.replace(/\n/g, '<br>');
        lyricsContent.innerHTML = '';
        lyricsContent.appendChild($p);
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    function getStationData(data, baseUrl, isMultiStream = false) {
      const name = isMultiStream ? data.multiradio_nombre : data.radio_nombre;
      const station = {
        name,
        hash: name.toLowerCase().replace(/\s/g, ''),
        description: isMultiStream ? data.multiradio_descripcion : data.radio_descripcion || data.radio_slogan,
        stream_url: isMultiStream ? data.multiradio_url : data.radio_url,
        tv_url: isMultiStream ? data.multiradio_tvstream : data.radio_video,
        cover: isMultiStream ? `${baseUrl}${data.multiradio_imagen}` : `${baseUrl}${data.radio_fondo}`,
        api: isMultiStream ? data.multiradio_api : data.radio_menu_nicio,
        server: 'spotify'
      };
      return station;
    }
    function setupSocial(container, data) {
      if (!container || !data) return;
      const socialItems = Object.keys(data).map(key => {
        return `<a href="${data[key]}" target="_blank" class="player-social-item player-social-${key}">
        ${icons[key]}
      </a>`;
      });
      container.innerHTML = socialItems.join('');
    }

    // Programacion de la radio
    /* eslint-disable camelcase */
    // Función para obtener el día actual en formato de 2 letras
    function getCurrentDay() {
      const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
      const date = new Date();
      const day = date.getDay();
      return days[day];
    }
    function compareNumbers(a, b) {
      return Number(a) === Number(b);
    }
    function sortByDay(programas) {
      const resultado = {
        monday: [],
        tuesday: [],
        wednesday: [],
        thursday: [],
        friday: [],
        saturday: [],
        sunday: []
      };
      programas.forEach(programa => {
        const {
          prog_horario_lu,
          prog_horario_ma,
          prog_horario_mi,
          prog_horario_ju,
          prog_horario_vi,
          prog_horario_sa,
          prog_horario_do,
          prog_lu,
          prog_ma,
          prog_mi,
          prog_ju,
          prog_vi,
          prog_sa,
          prog_do,
          prog_titulo,
          prog_descripcion,
          prog_foto
        } = programa;
        const item = {
          titulo: prog_titulo,
          descripcion: prog_descripcion,
          foto: prog_foto,
          horario: null
        };

        // Verificar cada día y agregar al arreglo correspondiente si está activo
        if (compareNumbers(prog_lu, 1)) {
          item.horario = prog_horario_lu;
          resultado.monday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ma, 1)) {
          item.horario = prog_horario_ma;
          resultado.tuesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_mi, 1)) {
          item.horario = prog_horario_mi;
          resultado.wednesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ju, 1)) {
          item.horario = prog_horario_ju;
          resultado.thursday.push({
            ...item
          });
        }
        if (compareNumbers(prog_vi, 1)) {
          item.horario = prog_horario_vi;
          resultado.friday.push({
            ...item
          });
        }
        if (compareNumbers(prog_sa, 1)) {
          item.horario = prog_horario_sa;
          resultado.saturday.push({
            ...item
          });
        }
        if (compareNumbers(prog_do, 1)) {
          item.horario = prog_horario_do;
          resultado.sunday.push({
            ...item
          });
        }
      });

      // Ordenar los programas por hora en cada día
      Object.keys(resultado).forEach(dia => {
        resultado[dia].sort((a, b) => a.horario.localeCompare(b.horario));
      });
      return resultado;
    }

    // Compara la hora actual con la hora de inicio de un programa
    // @param {string} time - Hora de inicio del programa
    // @returns {number} - 1 si la hora actual es mayor, -1 si la hora actual es menor, 0 si son iguales
    function compareTime(time) {
      const date = new Date();
      const currentHour = date.getHours();
      const currentMinutes = date.getMinutes();
      const [programHour, programMinutes] = time.split(':').map(Number);
      if (currentHour > programHour || currentHour === programHour && currentMinutes > programMinutes) {
        return 1; // Hora actual es mayor
      } else if (currentHour < programHour || currentHour === programHour && currentMinutes < programMinutes) {
        return -1; // Hora actual es menor
      } else {
        return 0; // Horas iguales
      }
    }
    function setupSchedule(data, baseUrl) {
      const programs = sortByDay(data);
      const days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
      const tabs = document.querySelectorAll('.program-button');
      const tabsElements = document.querySelectorAll('.nexo-schedule-list');
      tabs.forEach(tab => {
        tab.onclick = event => {
          const target = event.target;
          const tabId = target.dataset.tab;
          tabs.forEach(tab => {
            tab.classList.remove('is-active');
          });
          tabsElements.forEach(tab => {
            tab.classList.remove('is-active');
          });
          const ul = document.getElementById(tabId);
          ul.classList.add('is-active');
          target.classList.add('is-active');
        };
      });
      days.forEach(day => {
        const ul = document.getElementById(`program-${day}`);
        ul.innerHTML = '';
        programs[day].forEach(item => {
          const li = document.createElement('li');
          li.classList.add('nexo-schedule-item');
          li.innerHTML = `
        <div class="nexo-schedule-item-picture">
          <img src="${baseUrl}${item.foto}" alt="${item.titulo}">
        </div>
        <div class="nexo-schedule-item-content">
          <h3 class="nexo-schedule-item-title">${item.titulo}</h3>
          <p class="nexo-schedule-item-description">${item.horario} - ${item.descripcion}</p>
        </div>
      `;

          // Comparar la hora actual con la hora de inicio del programa
          const result = compareTime(item.horario);

          // Agregar la clase is-past si la hora actual es mayor
          // Solo agregar las clases si el programa es del día actual
          if (result > 0 && day === getCurrentDay()) {
            li.classList.add('is-past');
          }
          ul.appendChild(li);
        });

        // al ultimo elemento con la clase is-past agregarle la clase is-current
        // Solo si el día es el día actual
        const pastItems = document.querySelectorAll('.is-past');
        if (pastItems.length > 0 && day === getCurrentDay()) {
          pastItems[pastItems.length - 1].classList.add('is-current');
        }
      });

      // Mostrar el día actual
      const currentDay = getCurrentDay();

      // Mostrar el día actual
      const allTabs = document.querySelectorAll('.program-button');
      allTabs.forEach(tab => {
        tab.classList.remove('is-active');
      });
      const currentTab = document.querySelector(`[data-tab="program-${currentDay}"]`);
      currentTab.classList.add('is-active');
      const allLists = document.querySelectorAll('.nexo-schedule-list');
      allLists.forEach(list => {
        list.classList.remove('is-active');
      });
      const currentList = document.getElementById(`program-${currentDay}`);
      currentList.classList.add('is-active');
      const isCurrentElement = currentList.querySelector('.is-current');
      const stationCurrentProgram = {
        name: isCurrentElement?.querySelector('.nexo-schedule-item-title')?.textContent
      };
      createProgram(stationCurrentProgram);
    }
    const isWave = localStorage.getItem('isWave') === 'true' || false;
    const rhythm = document.querySelector('.player-button-wave');
    if (isWave) {
      visualizerContainer.classList.add('is-wave');
      rhythm.classList.add('is-wave');
    }
    rhythm.addEventListener('click', () => {
      rhythm.classList.toggle('is-wave');
      visualizerContainer.classList.toggle('is-wave');
      localStorage.setItem('isWave', visualizerContainer.classList.contains('is-wave'));
    });

    // Iniciar la aplicación
    function initApp() {
      // Variables para almacenar información que se actualizará
      let currentSongPlaying;
      let timeoutId;
      let timeoutIdProgram;
      let loadStations = [];
      const json = window.streams || {};
      const baseUrl = json.base_url || '';
      const urlServer = json.url_server || 'api/';
      const idUser = json.id_user || '';
      const isMultiStream = json.multi_stream || false;
      const radioPath = isMultiStream ? 'multiradio-app/' : 'radio-app/';
      const dataUrl = `${baseUrl}${urlServer}${radioPath}${idUser}`;
      const dataSocialUrl = `${baseUrl}${urlServer}radio-app/${idUser}`;
      const dataAppsUrl = `${baseUrl}${urlServer}radio-app/${idUser}`;
      const dataProgramUrl = `${baseUrl}${urlServer}programa-app/${idUser}`;
      fetch(dataUrl).then(response => response.json()).then(data => {
        if (isMultiStream) {
          const stations = data.map(station => getStationData(station, baseUrl, true));
          loadStations = stations;
          currentStation = loadStations[0];
          const stationButton = document.querySelector('.btn[data-outside="offcanvas-stations"]');
          if (stationButton) {
            stationButton.classList.add('is-loaded');
          }
        } else {
          const station = getStationData(data, baseUrl);
          currentStation = station;
        }

        // Establecer los assets de la página
        setAssetsInPage(currentStation);

        // Establecer la fuente de audio
        audio.src = currentStation.stream_url;

        // Iniciar el stream
        function init(current) {
          // Cancelar el timeout anterior
          if (timeoutId) clearTimeout(timeoutId);

          // Si la url de la estación actual es diferente a la estación actual, se actualiza la información
          if (currentStation.stream_url !== current.stream_url) {
            currentStation = current;
          }
          currentStation.server || 'itunes';
          const jsonUri = API_URL + encodeURIComponent(current.stream_url);
          fetch(jsonUri).then(response => response.json()).then(async song => {
            if (song.now_playing) {
              songProgress(song.now_playing.duration, song.now_playing.elapsed);
            }
            const title = song.title || song.song;
            const query = title.trim() !== '' ? `${song.artist} - ${title}` : `${current.name} - ${current.descripcion}`;
            if (currentSongPlaying !== query) {
              // Actualizar la canción actual
              currentSongPlaying = query;
              const songData = await getDataFromSearchApi(query, SERVICE);
              if (songData.artwork.includes('https://nexoapi.lat/') || songData.artwork === null || title.trim() === '') {
                songData.artwork = current.picture || current.cover;
                songData.title = current.name;
                songData.artist = current.description;
              } else {
                songData.title = title;
                songData.artist = song.artist;
              }
              normalizeHistory(song);

              // Establecer datos de la canción actual
              currentSong(songData, currentStation.cover);
              setMediaSession(songData, currentStation.cover);
              setHistory(song.song_history, currentStation);
              setLyrics(songData.artist, songData.title);
            }
          }).catch(error => console.log(error));
          timeoutId = setTimeout(() => {
            init(current);
          }, TIME_TO_REFRESH);
        }
        init(currentStation);

        // DATOS DE LA PROGRAMACIÓN
        function initProgram() {
          if (timeoutIdProgram) clearTimeout(timeoutIdProgram);
          fetch(dataProgramUrl).then(response => response.json()).then(data => {
            setupSchedule(data, baseUrl);
          }).catch(error => {
            console.error('Error:', error);
          });

          // Refrescar cada 1 minuto
          timeoutIdProgram = setTimeout(() => {
            initProgram();
          }, 60000);
        }
        initProgram();
        createStations(loadStations, currentStation, station => {
          init(station);
        });
        const nextStation = document.querySelector('.player-button-forward-step');
        const prevStation = document.querySelector('.player-button-backward-step');
        if (nextStation) {
          nextStation.addEventListener('click', () => {
            const next = stationsList.querySelector('.is-active').nextElementSibling;
            if (next) {
              next.click();
            }
          });
        }
        if (prevStation) {
          prevStation.addEventListener('click', () => {
            const prev = stationsList.querySelector('.is-active').previousElementSibling;
            if (prev) {
              prev.click();
            }
          });
        }
      });

      // Botones Apps
      fetch(dataAppsUrl).then(response => response.json()).then(data => {
        setupApps(getApps(data)); // Llama a la función setupApps con el resultado de getApps
      }).catch(error => {
        console.error('Error:', error);
      });
      function setupApps(apps) {
        // Verifica si hay URL para Android y actualiza el botón
        if (apps.android) {
          const androidButton = document.getElementById('androidButton');
          androidButton.href = apps.android; // Asigna la URL para Android
          androidButton.style.display = 'inline-block'; // Asegura que se muestre el botón
        } else {
          const androidButton = document.getElementById('androidButton');
          androidButton.style.display = 'none'; // Oculta el botón si no hay URL
        }

        // Verifica si hay URL para iOS y actualiza el botón
        if (apps.ios) {
          const iosButton = document.getElementById('iosButton');
          iosButton.href = apps.ios; // Asigna la URL para iOS
          iosButton.style.display = 'inline-block'; // Asegura que se muestre el botón
        } else {
          const iosButton = document.getElementById('iosButton');
          iosButton.style.display = 'none'; // Oculta el botón si no hay URL
        }
      }

      // Botones sociales
      fetch(dataSocialUrl).then(response => response.json()).then(data => {
        setupSocial(playerSocial, getSocials(data));
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    initApp();
    function setDeviceInfo() {
      if (navigator.userAgent.match(/Android/i)) {
        return `${icons.android} Instalar en Android`;
      }
      if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {
        return `${icons.apple} Instalar en iOS`;
      }
      return `${icons.windows} Instalar en Windows`;
    }

    // Función para instalar PWA en el navegador
    function installPWA() {
      const container = document.getElementById('install-container');
      const close = document.getElementById('close-install');
      const install = document.getElementById('install-pwa');
      const checkIosDevice = () => {
        const ua = navigator.userAgent;
        const vendor = navigator.vendor;
        const isSafari = /Safari/.test(ua) && /Apple Computer/.test(vendor) && !/CriOS/.test(ua); // Excluir Chrome

        return isSafari;
      };
      if (checkIosDevice()) return;
      if (!container) return;
      if (!install) return;
      close.addEventListener('click', () => {
        container.classList.remove('is-active');
      });
      install.innerHTML = setDeviceInfo();
      const isMobileDevice = () => {
        const mobileDeviceRegex = /Android|Windows Phone|Windows|iPhone|iPad|iPod|iOS|Macintosh/i;
        return mobileDeviceRegex.test(navigator.userAgent);
      };
      const isFirefox = () => {
        return navigator.userAgent.includes('Firefox');
      };
      let installPrompt = null;
      window.addEventListener('beforeinstallprompt', event => {
        event.preventDefault();
        installPrompt = event;
        if (isMobileDevice() && !isFirefox()) {
          container.classList.add('is-active');
        }
      });
      install.addEventListener('click', async () => {
        if (!installPrompt) {
          console.warn('No se puede instalar la PWA');
          return;
        }
        const result = await installPrompt.prompt();
        if (result.outcome === 'accepted') {
          console.warn('PWA instalada');
          container.classList.remove('is-active');
        } else {
          console.warn('PWA no instalada');
        }
      });
    }
    installPWA();
  }

  function videoplayer() {
    const buttons = document.querySelectorAll('[data-outside]');
    const ACTIVE_CLASS = 'is-active';
    const BACKGROUNDS = window?.streams?.background_header || {};

    // Random background
    function randomBackground() {
      const img = document.querySelector('.player-header-cover');
      if (!img) return;
      const keys = Object.keys(BACKGROUNDS);
      const randomKey = keys[Math.floor(Math.random() * keys.length)];
      const randomBackground = BACKGROUNDS[randomKey];
      img.src = randomBackground;
    }
    randomBackground();
    function outsideClick(button) {
      if (!button) return;
      const target = document.getElementById(button.dataset.outside);
      if (!target) return;
      function toggleClasses() {
        button.classList.toggle(ACTIVE_CLASS);
        target.classList.toggle(ACTIVE_CLASS);
        if (button.classList.contains(ACTIVE_CLASS)) {
          document.addEventListener('click', clickOutside);
          return;
        }
        document.removeEventListener('click', clickOutside);
      }
      button.addEventListener('click', toggleClasses);
      function clickOutside(event) {
        if (!target.contains(event.target) && !button.contains(event.target)) {
          toggleClasses();
          document.removeEventListener('click', clickOutside);
        }
      }
      const closeButton = target.querySelector('[data-close]');
      if (closeButton) {
        closeButton.addEventListener('click', () => {
          button.classList.remove(ACTIVE_CLASS);
          target.classList.remove(ACTIVE_CLASS);
          document.removeEventListener('click', clickOutside);
        });
      }
    }
    function initOutsideClick() {
      buttons.forEach(button => {
        outsideClick(button);
      });
    }
    const headerBar = document.querySelector('.header-bar');
    window.addEventListener('scroll', function () {
      if (window.scrollY > 0) {
        headerBar.classList.add('header-fixed');
      } else {
        headerBar.classList.remove('header-fixed');
      }
    });
    const videoModule = document.getElementById('player-video-module');
    window.addEventListener('scroll', function () {
      if (window.scrollY > videoModule.offsetTop + videoModule.offsetHeight) {
        videoModule.classList.add('has-fixed');
      } else {
        videoModule.classList.remove('has-fixed');
      }
    });

    // Función para inicializar el lienzo (canvas)
    function initCanvas(container) {
      const canvas = document.createElement('canvas');
      canvas.setAttribute('id', 'visualizerCanvas');
      canvas.setAttribute('class', 'visualizer-item');
      container.appendChild(canvas);
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
      return canvas;
    }

    // Función para cambiar el lienzo según el tamaño del contenedor
    function resizeCanvas(canvas, container) {
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
    }

    // Visualizer
    const visualizer = (audio, container) => {
      if (!audio || !container) {
        return;
      }
      const options = {
        fftSize: container.dataset.fftSize || 2048,
        numBars: container.dataset.bars || 40,
        maxHeight: 80
      };
      const ctx = new AudioContext();
      const audioSource = ctx.createMediaElementSource(audio);
      const analyzer = ctx.createAnalyser();
      audioSource.connect(analyzer);
      audioSource.connect(ctx.destination);
      const frequencyData = new Uint8Array(analyzer.frequencyBinCount);
      const canvas = initCanvas(container);
      const canvasCtx = canvas.getContext('2d');

      // Crear barras
      const renderBars = () => {
        resizeCanvas(canvas, container);
        analyzer.getByteFrequencyData(frequencyData);
        {
          analyzer.fftSize = options.fftSize;
        }
        canvasCtx.clearRect(0, 0, canvas.width, canvas.height);
        for (let i = 0; i < options.numBars; i++) {
          const index = Math.floor((i + 10) * (i < options.numBars / 2 ? 2 : 1));
          const fd = frequencyData[index];
          const barHeight = Math.max(4, fd / 255 * options.maxHeight);
          const barWidth = canvas.width / options.numBars;
          const x = i * barWidth;
          const y = canvas.height - barHeight;
          canvasCtx.fillStyle = 'white';
          canvasCtx.fillRect(x, y, barWidth - 2, barHeight);
        }
        requestAnimationFrame(renderBars);
      };
      renderBars();

      // Listener del cambio de espacio en la ventana
      window.addEventListener('resize', () => {
        resizeCanvas(canvas, container);
      });
    };

    /* eslint-disable camelcase */
    // Función para obtener el día actual en formato de 2 letras
    function getCurrentDay$1() {
      const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
      const date = new Date();
      const day = date.getDay();
      return days[day];
    }
    function compareNumbers(a, b) {
      return Number(a) === Number(b);
    }
    function sortByDay(programas) {
      const resultado = {
        monday: [],
        tuesday: [],
        wednesday: [],
        thursday: [],
        friday: [],
        saturday: [],
        sunday: []
      };
      programas.forEach(programa => {
        const {
          prog_horario_lu,
          prog_horario_ma,
          prog_horario_mi,
          prog_horario_ju,
          prog_horario_vi,
          prog_horario_sa,
          prog_horario_do,
          prog_lu,
          prog_ma,
          prog_mi,
          prog_ju,
          prog_vi,
          prog_sa,
          prog_do,
          prog_titulo,
          prog_descripcion,
          prog_foto
        } = programa;
        const item = {
          titulo: prog_titulo,
          descripcion: prog_descripcion,
          foto: prog_foto,
          horario: null
        };

        // Verificar cada día y agregar al arreglo correspondiente si está activo
        if (compareNumbers(prog_lu, 1)) {
          item.horario = prog_horario_lu;
          resultado.monday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ma, 1)) {
          item.horario = prog_horario_ma;
          resultado.tuesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_mi, 1)) {
          item.horario = prog_horario_mi;
          resultado.wednesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ju, 1)) {
          item.horario = prog_horario_ju;
          resultado.thursday.push({
            ...item
          });
        }
        if (compareNumbers(prog_vi, 1)) {
          item.horario = prog_horario_vi;
          resultado.friday.push({
            ...item
          });
        }
        if (compareNumbers(prog_sa, 1)) {
          item.horario = prog_horario_sa;
          resultado.saturday.push({
            ...item
          });
        }
        if (compareNumbers(prog_do, 1)) {
          item.horario = prog_horario_do;
          resultado.sunday.push({
            ...item
          });
        }
      });

      // Ordenar los programas por hora en cada día
      Object.keys(resultado).forEach(dia => {
        resultado[dia].sort((a, b) => a.horario.localeCompare(b.horario));
      });
      return resultado;
    }

    // Compara la hora actual con la hora de inicio de un programa
    // @param {string} time - Hora de inicio del programa
    // @returns {number} - 1 si la hora actual es mayor, -1 si la hora actual es menor, 0 si son iguales
    function compareTime(time) {
      const date = new Date();
      const currentHour = date.getHours();
      const currentMinutes = date.getMinutes();
      const [programHour, programMinutes] = time.split(':').map(Number);
      if (currentHour > programHour || currentHour === programHour && currentMinutes > programMinutes) {
        return 1; // Hora actual es mayor
      } else if (currentHour < programHour || currentHour === programHour && currentMinutes < programMinutes) {
        return -1; // Hora actual es menor
      } else {
        return 0; // Horas iguales
      }
    }
    function setupSchedule(data, baseUrl) {
      const programs = sortByDay(data);
      const days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
      const tabs = document.querySelectorAll('.program-button');
      const tabsElements = document.querySelectorAll('.nexo-schedule-list');
      tabs.forEach(tab => {
        tab.onclick = event => {
          const target = event.target;
          const tabId = target.dataset.tab;
          tabs.forEach(tab => {
            tab.classList.remove('is-active');
          });
          tabsElements.forEach(tab => {
            tab.classList.remove('is-active');
          });
          const ul = document.getElementById(tabId);
          ul.classList.add('is-active');
          target.classList.add('is-active');
        };
      });
      days.forEach(day => {
        const ul = document.getElementById(`program-${day}`);
        ul.innerHTML = '';
        programs[day].forEach(item => {
          const li = document.createElement('li');
          li.classList.add('nexo-schedule-item');
          li.innerHTML = `
        <div class="nexo-schedule-item-picture">
          <img src="${baseUrl}${item.foto}" alt="${item.titulo}">
        </div>
        <div class="nexo-schedule-item-content">
          <h3 class="nexo-schedule-item-title">${item.titulo}</h3>
          <p class="nexo-schedule-item-description">${item.horario} - ${item.descripcion}</p>
        </div>
      `;

          // Comparar la hora actual con la hora de inicio del programa
          const result = compareTime(item.horario);

          // Agregar la clase is-past si la hora actual es mayor
          // Solo agregar las clases si el programa es del día actual
          if (result > 0 && day === getCurrentDay$1()) {
            li.classList.add('is-past');
          }
          ul.appendChild(li);
        });

        // al ultimo elemento con la clase is-past agregarle la clase is-current
        // Solo si el día es el día actual
        const pastItems = document.querySelectorAll('.is-past');
        if (pastItems.length > 0 && day === getCurrentDay$1()) {
          pastItems[pastItems.length - 1].classList.add('is-current');
        }
      });

      // Mostrar el día actual
      const currentDay = getCurrentDay$1();

      // Mostrar el día actual
      const alltabs = document.querySelectorAll('.program-button');
      alltabs.forEach(tab => {
        tab.classList.remove('is-active');
      });
      const currentTab = document.querySelector(`[data-tab="program-${currentDay}"]`);
      currentTab.classList.add('is-active');
      const allLists = document.querySelectorAll('.nexo-schedule-list');
      allLists.forEach(list => {
        list.classList.remove('is-active');
      });
      const currentList = document.getElementById(`program-${currentDay}`);
      currentList.classList.add('is-active');
    }

    // Iconos de Meteor Icons: https://meteoricons.com/
    const icons = {
      play: '<svg width="24"  height="24" viewBox="0 0 24 24"  fill="currentColor"><path stroke="none" d="M0 0h24v24H0z" fill="none"/><path d="M6 4v16a1 1 0 0 0 1.524 .852l13 -8a1 1 0 0 0 0 -1.704l-13 -8a1 1 0 0 0 -1.524 .852z" /></svg>',
      pause: '<svg width="24"  height="24" viewBox="0 0 24 24"  fill="currentColor"><path stroke="none" d="M0 0h24v24H0z" fill="none"/><path d="M9 4h-2a2 2 0 0 0 -2 2v12a2 2 0 0 0 2 2h2a2 2 0 0 0 2 -2v-12a2 2 0 0 0 -2 -2z" /><path d="M17 4h-2a2 2 0 0 0 -2 2v12a2 2 0 0 0 2 2h2a2 2 0 0 0 2 -2v-12a2 2 0 0 0 -2 -2z" /></svg>',
      stop: '<svg width="24"  height="24" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M6 5H18C18.5523 5 19 5.44772 19 6V18C19 18.5523 18.5523 19 18 19H6C5.44772 19 5 18.5523 5 18V6C5 5.44772 5.44772 5 6 5Z"/></svg>'
    };
    const svgIcons = {
      play: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M10 16.5v-9l6 4.5M12 2A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>',
      pause: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M15 16h-2V8h2m-4 8H9V8h2m1-6A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>',
      facebook: '<svg class="i i-facebook" viewBox="0 0 24 24"><path d="M17 14h-3v8h-4v-8H7v-4h3V7a5 5 0 0 1 5-5h3v4h-3q-1 0-1 1v3h4Z"></path></svg>',
      youtube: '<svg class="i i-youtube" viewBox="0 0 24 24"><path d="M1.5 17q-1-5.5 0-10Q1.9 4.8 4 4.5q8-1 16 0 2.1.3 2.5 2.5 1 4.5 0 10-.4 2.2-2.5 2.5-8 1-16 0-2.1-.3-2.5-2.5Zm8-8.5v7l6-3.5Z"></path></svg>',
      instagram: '<svg class="i i-instagram" viewBox="0 0 24 24"><circle cx="12" cy="12" r="4"></circle><rect width="20" height="20" x="2" y="2" rx="5"></rect><path d="M17.5 6.5h0"></path></svg>',
      whatsapp: '<svg class="i i-whatsapp" viewBox="0 0 24 24"><circle cx="9" cy="9" r="1"></circle><circle cx="15" cy="15" r="1"></circle><path d="M8 9a7 7 0 0 0 7 7m-9 5.2A11 11 0 1 0 2.8 18L2 22Z"></path></svg>',
      tiktok: '<svg class="i i-tiktok" viewBox="0 0 24 24"><path d="M22 6v5q-4 0-6-2v7a7 7 0 1 1-5-6.7m0 6.7a2 2 0 1 0-2 2 2 2 0 0 0 2-2V1h5q2 5 6 5"></path></svg>',
      bluesky: '<svg class="i i-bluesky" viewBox="0 0 24 24"><path d="M12 10Q2-2 2 6t5 8q-5 3-1 6t6-3q2 6 6 3t-1-6q5 0 5-8t-10 4"></path></svg>',
      discord: '<svg class="i i-discord" viewBox="0 0 24 24"><path d="M9 3q-2.5.5-5 2-3 5-3 12 2 2.5 6 4 1-1.5 1.5-3.5M7 17q5 2 10 0m-1.5.5q.5 2 1.5 3.5 4-1.5 6-4 0-7-3-12-2.5-1.5-5-2l-1 2q-2-.5-4 0L9 3"></path><circle cx="8" cy="12" r="1"></circle><circle cx="16" cy="12" r="1"></circle></svg>',
      soundcloud: '<svg class="i i-soundcloud" viewBox="0 0 24 24"><path d="M11 7v11m-3 0v-7m-6 6v-3m3-4v8m9 0h6a1 1 0 0 0 0-6 6 6 0 0 0-6-6Z"></path></svg>',
      threads: '<svg class="i i-threads" viewBox="0 0 24 24"><path d="M20 8C18 0 6 0 4 8s1.5 14 8 14 10-7.6 4-10-9 2-7 4 7 1 7-4-5-6-7-4"></path></svg>',
      twitter: '<svg class="i i-twitter" viewBox="0 0 24 24"><path d="M12 7.5a4.5 4.5 0 0 1 8-3Q22 4 23 3q0 2-2 4A13 13 0 0 1 1 19q5 0 7-2-8-4-5-13 4 5 9 5Z"></path></svg>',
      x: '<svg class="i i-x" viewBox="0 0 24 24"><path d="m3 21 7.5-7.5m3-3L21 3M8 3H3l13 18h5Z"></path></svg>',
      deezer: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill-rule="evenodd" d="M20.1 3.6c.2-1.2.6-2 1-2 .6 0 1.1 2.8 1.1 6.2 0 3.5-.5 6.3-1.2 6.3-.3 0-.5-.5-.7-1.3-.3 3-1 5-1.8 5-.6 0-1.1-1.3-1.5-3.2-.3 3.7-.9 6.2-1.6 6.2-.5 0-.9-1-1.2-2.7C13.8 21.6 13 24 12 24c-1 0-1.9-2.3-2.3-5.8-.3 1.7-.7 2.7-1.1 2.7-.8 0-1.4-2.6-1.6-6.2-.4 2-1 3.1-1.5 3.1-.8 0-1.5-2-1.8-4.9-.2.8-.5 1.3-.8 1.3-.6 0-1.2-2.8-1.2-6.3 0-3.4.6-6.2 1.2-6.2.4 0 .7.8 1 2C4.1 1.5 4.8 0 5.4 0c.7 0 1.4 2 1.7 5 .4-2.2.8-3.5 1.4-3.5.7 0 1.4 2.7 1.6 6.4.4-1.9 1-3.1 1.8-3.1.7 0 1.4 1.2 1.8 3.1.2-3.7.9-6.4 1.6-6.4.5 0 1 1.3 1.3 3.5.3-3 1-5 1.8-5 .7 0 1.2 1.4 1.6 3.6zM.7 10C.3 10 0 8.7 0 7.2c0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8S1.1 10 .7 10zm22.6 0c-.4 0-.7-1.3-.7-2.8 0-1.5.3-2.8.7-2.8.4 0 .7 1.3.7 2.8s-.3 2.8-.7 2.8z"/></svg>'
    };
    const pixel = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAAAXNSR0IArs4c6QAAAA1JREFUGFdj+P//PxcACQYDCF0ysWYAAAAASUVORK5CYII=';

    // Convertir Unix timestamp a fecha
    function unixToDate(unix) {
      return new Date(unix * 1000);
    }

    // Convertir a tiempo relativo
    function timeAgo(date) {
      const getSecondsDiff = timestamp => (Date.now() - timestamp) / 1000;

      // Unidades de tiempo
      const DATE_UNITS = {
        day: 86400,
        hour: 3600,
        minute: 60,
        second: 1
      };

      // Obtener unidad y valor de la fecha
      const getUnitAndValueDate = secondsElapsed => {
        for (const [unit, secondsInUnit] of Object.entries(DATE_UNITS)) {
          if (secondsElapsed >= secondsInUnit || unit === 'second') {
            const value = Math.floor(secondsElapsed / secondsInUnit) * -1;
            return {
              value,
              unit
            };
          }
        }
      };

      // Obtener tiempo relativo
      const getTimeAgo = timestamp => {
        const rtf = new Intl.RelativeTimeFormat();
        const secondsElapsed = getSecondsDiff(timestamp);
        const {
          value,
          unit
        } = getUnitAndValueDate(secondsElapsed);
        return rtf.format(value, unit);
      };

      // Resultado
      const reference = new Date(date);
      return getTimeAgo(reference);
    }

    // Obtener porcentaje entre dos valores
    function getPercentage(value, total) {
      return value * 100 / total;
    }
    function convertSecondsToMinutes(seconds) {
      const minutes = Math.floor(seconds / 60);
      const remainingSeconds = seconds % 60;
      return `${minutes}:${remainingSeconds.toFixed(0) < 10 ? '0' : ''}${remainingSeconds.toFixed(0)}`;
    }

    // Crear un elemento HTML a partir de una cadena de texto
    function createElementFromHTML(htmlString) {
      const div = document.createElement('div');
      div.innerHTML = htmlString.trim();
      return div.firstChild;
    }
    function createTempImage(src) {
      return new Promise((resolve, reject) => {
        const img = document.createElement('img');
        img.crossOrigin = 'Anonymous';
        img.src = `https://images.weserv.nl/?url=${src}`;
        img.onload = () => resolve(img);
        img.onerror = reject;
      });
    }
    function getSocials(data) {
      const socials = {};
      if (data.radio_facebook_url) {
        socials.facebook = data.radio_facebook_url;
      }
      if (data.radio_youtube) {
        socials.youtube = data.radio_youtube;
      }
      if (data.radio_instagram) {
        socials.instagram = data.radio_instagram;
      }
      if (data.radio_whatsapp) {
        socials.whatsapp = `https://wa.me/+${data.radio_whatsapp}`;
      }
      if (data.radio_menu_noticias) {
        socials.tiktok = data.radio_menu_noticias;
      }
      return socials;
    }
    function getApps(data) {
      const apps = {};
      // Verifica si existe el campo para la app Android
      if (data.radio_fondo_color) {
        apps.android = data.radio_fondo_color; // Asigna la URL para la app Android
      }
      // Verifica si existe el campo para la app iOS
      if (data.radio_splash_color) {
        apps.ios = data.radio_splash_color; // Asigna la URL para la app iOS
      }
      return apps;
    }

    // Devolver una promesa para saber si la imagen se ha cargado correctamente
    // @param {string} src - URL de la imagen
    // @returns {Promise} - Promesa que se resuelve si la imagen se carga correctamente
    function loadImage(src) {
      return new Promise((resolve, reject) => {
        const img = new Image();
        img.onload = () => resolve(img);
        img.onerror = reject;
        img.src = src;
      });
    }

    // Agrega una transición de deslizamiento a la imagen
    // @param {HTMLElement} container - Contenedor de la imagen
    // @param {string} src - URL de la imagen
    function slideUpImageTransition(container, src) {
      const img = document.createElement('img');
      const size = container.clientHeight;
      img.src = src;
      img.width = size;
      img.height = size;
      container.appendChild(img);
      const firstImg = container.querySelector('img:first-child');
      firstImg.style.marginLeft = `-${size}px`;
      firstImg.addEventListener('transitionend', () => {
        const allImgExcLast = container.querySelectorAll('img:not(:last-child)');
        allImgExcLast.forEach(img => img.remove());
      }, {
        once: true
      });
    }

    // Agrega una transición de desvanecimiento a la imagen
    // @param {HTMLElement} container - Contenedor de la imagen
    // @param {string} src - URL de la imagen
    function fadeImageTransition(container, src) {
      container.style.opacity = 0;
      container.addEventListener('transitionend', () => {
        container.src = src;
        container.style.opacity = 1;
        container.addEventListener('transitionend', () => {
          container.removeAttribute('style');
        }, {
          once: true
        });
      });
    }
    function setAccentColor(image, colorThief) {
      const dom = document.querySelector('html');
      const metaThemeColor = document.querySelector('meta[name=theme-color]');
      if (image.complete) {
        dom.setAttribute('style', `--accent: rgb(${colorThief.getColor(image)})`);
        metaThemeColor.setAttribute('content', `rgb(${colorThief.getColor(image)})`);
      } else {
        image.addEventListener('load', function () {
          dom.setAttribute('style', `--accent: rgb(${colorThief.getColor(image)})`);
          metaThemeColor.setAttribute('content', `rgb(${colorThief.getColor(image)})`);
        });
      }
    }

    // Set attributes
    // @param {HTMLElement} element - element
    // @param {Object} attributes - attributes
    // @returns {void}
    const setAttributes = (element, attributes) => {
      for (const key in attributes) {
        element.setAttribute(key, attributes[key]);
      }
    };

    // Create embed
    // @param {string} url - url
    // @param {Object} params - attributes
    // @returns {HTMLIFrameElement} - iframe element
    function createEmbed({
      url,
      params = {}
    }) {
      const $iframe = document.createElement('iframe');
      $iframe.src = url;
      $iframe.frameborder = 0;
      setAttributes($iframe, params);
      return $iframe;
    }

    // Get video info
    // @param {string} url - url
    // @returns {Promise<string>} - title
    async function getVideoInfo(videoId) {
      const data = await fetch(`https://noembed.com/embed?dataType=json&url=https://www.youtube.com/watch?v=${videoId}`);
      return data.json();
    }
    const modal = document.querySelector('.modal');
    const modalBody = modal.querySelector('.modal-body');
    const OPEN_CLASS = 'is-open';
    let currentVideo$1 = null;
    function initVideoPlayer(videoUrl) {
      const $video = document.createElement('video');
      $video.id = 'player';
      $video.classList.add('video-js', 'video-js-stream');
      modalBody.innerHTML = '';
      modalBody.appendChild($video);
      const videoId = document.getElementById('player');

      // eslint-disable-next-line no-undef
      currentVideo$1 = videojs(videoId, {
        sources: [{
          src: videoUrl,
          type: 'application/x-mpegURL'
        }],
        controls: true,
        autoplay: true
      });
    }
    function createShareButton(name, url) {
      const $a = document.createElement('a');
      $a.className = `player-social-link player-social-link-${name}`;
      $a.href = url;
      $a.target = '_blank';
      $a.innerHTML = svgIcons[name];
      return $a;
    }
    function initShareModal({
      artist,
      title,
      artwork
    }) {
      const shareUrl = window.location.href;
      const shareText = `Estoy escuchando ${title} de ${artist} en ${shareUrl}`;
      modalBody.innerHTML = `
    <h2 class="fw-800 mb-0.75">COMPARTIR</h2>
    <img src="${artwork}" alt="portada modal" class="player-modal-image" width="200" height="200">
    <div class="fs-4 fw-500 song-name">${title}</div>
    <span class="fs-5">${artist}</span>
    <div class="flex g-0.5 justify-center modal-social player-socials wrap">
      ${createShareButton('facebook', `https://www.facebook.com/sharer/sharer.php?u=${shareUrl}`).outerHTML}
      ${createShareButton('x', `https://twitter.com/intent/tweet?text=${encodeURIComponent(shareText)}`).outerHTML}
      ${createShareButton('whatsapp', `https://api.whatsapp.com/send?text=${encodeURIComponent(shareText)}`).outerHTML}
    </div>
  `;
    }
    function createForm(formAction, formId) {
      modalBody.innerHTML = `
    <form id="${formId}" action="${formAction}" method="POST" class="modal-form flex column g-1">
      <input type="text" name="name" placeholder="Nombre" required>
      <input type="email" name="email" placeholder="Correo" required>
      <input type="tel" name="contact" placeholder="Teléfono" required>
      <textarea name="message" placeholder="Mensaje" required></textarea>
      <input type="hidden" name="promo" value="${formId}">
      <button type="submit" class="btn btn-full btn-input">Enviar</button>
    </form>
  `;
      const $form = modalBody.querySelector('form');
      $form.addEventListener('submit', async e => {
        e.preventDefault();
        const formData = new FormData($form);
        const response = await fetch(formAction, {
          method: 'POST',
          body: formData
        });
        document.body.classList.add(response.ok ? 'modal-success' : 'modal-error');
        modal.classList.add('modal-message');
        setTimeout(() => {
          document.body.classList.remove('modal-success', 'modal-error');
          modal.classList.remove('modal-message');
        }, 3000);
        if (response.ok) $form.reset();
      });
    }
    function openLyricsModal(lyrics) {
      modalBody.innerHTML = `
    <div class="lyrics-header">
      <h2 class="modal-lyrics-title m:fs-3 fs-5 fw-500">Letra</h2>
    </div>
    <pre>${lyrics || 'No se encontraron letras'}</pre>
  `;
    }
    function openSongModal({
      artist,
      title,
      artwork,
      genre,
      album
    }) {
      modalBody.innerHTML = `<div class="modal-song-info p-1 text-center">
    <img src="${artwork}" alt="portada modal" class="player-modal-image" width="200" height="200">
    <h3 class="fs-4 fw-700 color-text-inverse song-name">${title}</h3>
    <div class="flex column color-text-inverse justify-center modal-social player-socials wrap">
      <span class="fs-5"><strong class="color-text-inverse fw-700">Artista: </strong>${artist}</span>
      <span class="fs-5"><strong class="color-text-inverse fw-700">Genero: </strong>${genre}</span>
      <span class="fs-5"><strong class="color-text-inverse fw-700">Album: </strong>${album}</span>
    </div>
  </div>
  `;
    }
    function openFullModal({
      titulo,
      contenido,
      imagen,
      time
    }) {
      document.body.classList.add('modal-article');
      modalBody.innerHTML = `
    <div class="modal-article-title">
      <h2 class="fs-5 fw-800 m:fs-2">${titulo}</h2>
      <span class="fs-5">${time}</span>
    </div>
    <div class="modal-article-content">
      ${contenido}
    </div>
  `;
    }
    function openModal(type, options = {}) {
      if (!modal) return;
      modal.classList.add(OPEN_CLASS);
      modalBody.innerHTML = '';
      modal.classList.add(`modal-${type}`);
      switch (type) {
        case 'video':
          initVideoPlayer(options.url);
          break;
        case 'share':
          initShareModal(options.data);
          break;
        case 'form':
          createForm(options.formAction, options.formId);
          break;
        case 'lyrics':
          openLyricsModal(options.lyrics);
          break;
        case 'song':
          openSongModal(options.data);
          break;
        case 'article':
          openFullModal(options.data);
          break;
        default:
          console.warn('Tipo de modal desconocido:', type);
          return;
      }
      function closeModal() {
        modal.classList.remove(OPEN_CLASS, `modal-${type}`);
        document.body.classList.remove('modal-article');
        if (currentVideo$1) currentVideo$1.dispose();
        modalBody.innerHTML = '';
        if (options.hook && typeof options.hook === 'function') {
          options.hook();
        }
      }
      modal.querySelector('.modal-close').addEventListener('click', closeModal);
      modal.addEventListener('click', e => {
        if (!modal.querySelector('.modal-content').contains(e.target)) closeModal();
      });
    }
    let currentIframeActive = null;
    const cache$1 = localStorage.getItem('ytListMini') ? JSON.parse(localStorage.getItem('ytListMini')) : {};
    async function fetchVideoList(data, CONTAINER_LIST) {
      // max 4 videos
      const items = data.length > 5 ? data.slice(0, 5) : data;
      for (const item of items) {
        const videoInfo = cache$1[item.video_url] || (await getVideoInfo(item.video_url));
        const $li = createVideoListItem(item.video_url, videoInfo);
        CONTAINER_LIST.appendChild($li);
        addEventListeners($li, item.video_url);
        cache$1[item.video_url] = videoInfo;
        localStorage.setItem('ytListMini', JSON.stringify(cache$1));
      }
      const $firstVideo = CONTAINER_LIST.querySelector('.ytlistmini-load-iframe');
      if ($firstVideo) {
        $firstVideo.click();
      }
    }
    function createVideoListItem(videoId, videoInfo) {
      const $div = document.createElement('div');
      const videoThumbnail = videoInfo.thumbnail_url.replace('hqdefault', 'mqdefault');
      $div.className = 'ytlistmini-item';
      $div.innerHTML = `
    <div class="ytlistmini-content" data-video-id="${videoId}">
      <div class="ytlistmini-element">
        <img src="${videoThumbnail}" alt="${videoInfo.title}">
        <div class="ytlistmini-info">
          <h3>${videoInfo.title}</h3>
          <p>${videoInfo.author_name}</p>
        </div>
        <button class="ytlistmini-load-iframe"></button>
      </div>
      <div class="ytlistmini-iframe"></div>
    </div>
  `;
      return $div;
    }
    function addEventListeners($li, videoId) {
      $li.querySelector('.ytlistmini-load-iframe').addEventListener('click', function () {
        handleLoadIframeButtonClick($li, videoId);
      });
    }
    function handleLoadIframeButtonClick($li, videoId) {
      const $iframe = createEmbed({
        url: `https://www.youtube.com/embed/${videoId}`,
        params: {
          allowfullscreen: '1',
          class: 'ytube-embed'
        }
      });
      if ($li.classList.contains('is-active')) return;
      $li.classList.add('is-active');
      if (currentIframeActive) {
        currentIframeActive.innerHTML = '';
        currentIframeActive.classList.remove('is-active');
        currentIframeActive.parentElement.parentElement.classList.remove('is-active');
      }
      const $iframeContainer = $li.querySelector('.ytlistmini-iframe');
      $iframeContainer.innerHTML = '';
      $iframeContainer.appendChild($iframe);
      currentIframeActive = $iframeContainer;
    }

    // Obtener los datos de la búsqueda
    // @param {string} query - Consulta de búsqueda
    // @param {string} service - Servicio de búsqueda
    // @returns {Promise<object>} - Datos de la búsqueda
    const cache = {};
    const getDataFromSearchApi = async (query, service) => {
      if (cache[query]) {
        return cache[query];
      }
      const streamUrl = `api/search.php?query=${encodeURIComponent(query)}&service=${service}`;
      const response = await fetch(streamUrl);
      const data = await response.json();

      // Si no responde
      if (!data.results) {
        return {};
      }
      const results = data.results;
      cache[query] = results;
      return results;
    };

    // Obtener letras de canciones usando Lyrics.ovh
    const getLyricsOvh = async (artist, name) => {
      try {
        const response = await fetch(`https://api.lyrics.ovh/v1/${encodeURIComponent(artist)}/${encodeURIComponent(name)}?auto=1`);
        if (!response.ok) throw new Error('No se pudo obtener la letra');
        const data = await response.json();
        if (data.lyrics) {
          return data.lyrics;
        } else {
          return 'Letra no disponible';
        }
      } catch (error) {
        console.warn('No se pudo obtener la letra de Lyrics.ovh');
        return 'Letra no disponible';
      }
    };

    // obtener letras de vagalume
    const getVagalumeLyrics = async (artist, name) => {
      const apikey = '1637b78dc3b129e6843ed674489a92d0';
      try {
        const response = await fetch(`https://api.vagalume.com.br/search.php?apikey=${apikey}&art=${encodeURIComponent(artist)}&mus=${encodeURIComponent(name)}`);
        if (!response.ok) throw new Error('No se pudo obtener la letra');
        const data = await response.json();
        if (data.type === 'notfound') {
          return 'Letra no disponible';
        }
        return data.mus[0].text;
      } catch (error) {
        console.warn('No se pudo obtener la letra de Vagalume');
        return 'Letra no disponible';
      }
    };

    // obtener letras
    const getLyrics = async (artist, name) => {
      let lyrics = await getLyricsOvh(artist, name);
      if (!lyrics || lyrics === 'Letra no disponible') {
        lyrics = await getVagalumeLyrics(artist, name);
      }
      return lyrics;
    };
    const TIMEOUT = window?.streams?.sliderTiming || 5000;

    // Establecer los elementos de la estación
    // @param {HTMLElement} container - Contenedor de la estación
    // @param {Object} data - Datos de la estación
    function setupStream(container, data) {
      if (!container || !data) return;
      const pictureEl = container.querySelector('.player-picture img:first-child');
      const backgroundEl = container.querySelector('.player-bg');
      // const backgroundHeaderEl = container.querySelector('.player-header-cover')
      const stationName = container.querySelector('.station-name');
      const streamName = container.querySelector('.stream-name');
      const streamDescription = container.querySelector('.stream-description');
      const stationDescription = container.querySelector('.station-description');
      const stationTv = container.querySelector('.station-tv');
      const stationPicture = container.querySelector('.station-picture img');
      if (pictureEl) {
        // eslint-disable-next-line no-undef
        const colorThief = new ColorThief();
        createTempImage(data.picture).then(img => {
          setAccentColor(img, colorThief);
        });
        pictureEl.src = data.picture;
      }
      if (backgroundEl) {
        backgroundEl.src = data.background;
      }

      // if (backgroundHeaderEl) {
      //   backgroundHeaderEl.src = data.background
      // }

      if (stationName) {
        stationName.textContent = data.name;
      }
      if (streamName) {
        streamName.textContent = data.name;
      }
      if (streamDescription) {
        streamDescription.textContent = data.description;
      }
      if (stationDescription) {
        stationDescription.textContent = data.description;
      }
      if (stationTv) {
        stationTv.classList.remove('none');
      }
      if (stationPicture) {
        stationPicture.src = data.picture;
      }
    }

    // Obtener el valor que corresponde al dia de la semana
    function getDayValue(item) {
      const day = new Date().getDay();
      switch (day) {
        case 1:
          return item.prog_horario_lu;
        case 2:
          return item.prog_horario_ma;
        case 3:
          return item.prog_horario_mi;
        case 4:
          return item.prog_horario_ju;
        case 5:
          return item.prog_horario_vi;
        case 6:
          return item.prog_horario_sa;
        case 0:
          return item.prog_horario_do;
        default:
          return '';
      }
    }

    // Función para obtener el día actual en formato de 2 letras
    function getCurrentDay() {
      const weekDays = ['do', 'lu', 'ma', 'mi', 'ju', 'vi', 'sa'];
      const day = new Date().getDay(); // 0 es domingo, 1 es lunes, etc.
      return weekDays[day];
    }
    function filterScheduleByDay(programs) {
      const currentDay = getCurrentDay();

      // Definir las claves de días a eliminar
      const clavesDias = ['prog_lu', 'prog_ma', 'prog_mi', 'prog_ju', 'prog_vi', 'prog_sa', 'prog_do'];
      const clavesHorario = ['prog_horario_lu', 'prog_horario_ma', 'prog_horario_mi', 'prog_horario_ju', 'prog_horario_vi', 'prog_horario_sa', 'prog_horario_do'];
      programs.filter(program => program['prog_' + currentDay] === '1');

      // Eliminar las claves de horario que no correspondan al día actual
      programs.forEach(program => {
        clavesHorario.forEach(clave => {
          if (clave !== 'prog_horario_' + currentDay) {
            delete program[clave];
          }
        });
      });

      // Eliminar los elementos del arreglo que no tengan horario
      programs = programs.filter(program => program['prog_horario_' + currentDay]);

      // Eliminar todas las claves de días
      programs.forEach(program => {
        clavesDias.forEach(clave => {
          delete program[clave];
        });
      });

      // Ordenar los programas por hora de menor a mayor
      programs.sort((a, b) => {
        const timeA = a['prog_horario_' + currentDay];
        const timeB = b['prog_horario_' + currentDay];
        return parseTime(timeA) - parseTime(timeB);
      });
      return programs;
    }

    // Función para convertir una hora "HH:mm" en minutos desde la medianoche
    function parseTime(time) {
      const [hours, minutes] = time.split(':').map(Number);
      return hours * 60 + minutes;
    }

    // Compara dos horas en formato HH:MM.
    // @param {string} hora1 - La primera hora.
    // @param {string} hora2 - La segunda hora.
    // @returns {string} - Indica si la primera hora es mayor, menor o igual a la segunda.
    function compareHours(hora, type, test) {
      // Obtener la hora actual, formateada en 24 horas, Ingles
      const currentTime = new Date().toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: false
      });

      // Fecha base arbitraria para crear objetos Date
      const base = '1970-01-01';
      const current = new Date(`${base}T${currentTime}:00`);
      const target = new Date(`${base}T${hora}:00`);
      if (type === 'greaterOrEqual') {
        return target >= current;
      }
      if (type === 'less') {
        return target < current;
      }
      if (type === 'equal') {
        return target === current;
      }
      return false;
    }
    function setupMiniSchedule(container, data, baseUrl) {
      const programList = document.getElementById('program-list');
      if (!container || !data.length) return;
      const programs = filterScheduleByDay(data);
      programList.innerHTML = '';
      programs.forEach(item => {
        const li = document.createElement('li');
        li.classList.add('schedule-item', 'flex', 'g-1', 'items-center');
        li.innerHTML = `
      <div class="schedule-item-picture flex-none">
        <img src="${baseUrl}${item.prog_foto}" alt="${item.prog_titulo}">
      </div>
      <div class="schedule-item-content truncate">
        <h3 class="schedule-item-title truncate">${item.prog_titulo}</h3>
        <p class="schedule-item-description truncate">${getDayValue(item)} - ${item.prog_descripcion}</p>
      </div>
    `;
        programList.appendChild(li);
      });
      container.innerHTML = '';

      // Ordenar la programación por hora en formato 24 horas
      programs.sort((a, b) => {
        const aTime = getDayValue(a);
        const bTime = getDayValue(b);
        return aTime.localeCompare(bTime);
      });
      const filteredPrograms = [];
      for (let i = programs.length - 1; i >= 0; i--) {
        const programTime = getDayValue(programs[i]);

        // const test = '12:01'

        if (compareHours(programTime, 'greaterOrEqual')) {
          filteredPrograms.unshift(programs[i]);
        }
        if (compareHours(programTime, 'less')) {
          filteredPrograms.unshift(programs[i]);
          break;
        }
      }
      filteredPrograms.slice(0, 2).forEach((item, index) => {
        const li = document.createElement('li');
        li.classList.add('schedule-item', 'flex', 'g-1');
        li.innerHTML = `
      <div class="schedule-item-picture flex-none">
        <img src="${baseUrl}${item.prog_foto}" alt="${item.prog_titulo}">
      </div>
      <div class="schedule-item-content truncate">
        ${index === 0 ? '<span class="player-program-badge"><i></i> al aire</span>' : ''}
        ${index === 1 ? '<span class="player-program-text"><i></i> a continuacion</span>' : ''}
        <h3 class="schedule-item-title truncate">${item.prog_titulo}</h3>
        <p class="schedule-item-description truncate">${getDayValue(item)} - ${item.prog_descripcion}</p>
      </div>
    `;
        container.appendChild(li);
      });
    }

    // Establecer el slider
    // @param {HTMLElement} container - Contenedor del slider
    // @param {Array} data - Datos de las imágenes
    // @param {string} baseUrl - URL base de las imágenes
    function setupSlider(container, data, baseUrl) {
      const banners = data.map(item => {
        return `<div class="slider-slide">
      <a href="${item.slide_url}" target="_blank">
        <img src="${baseUrl}${item.slide_foto}" alt="Slide" class="slider-image" width="100%">
      </a>
    </div>`;
      });
      container.innerHTML = banners.join('');
      container.classList.add('is-loaded');
      if (container) {
        const SLIDES = data.length;

        // Mover el contenedor Wrapper dependiendo de la cantidad de slides
        let currentSlide = 0;
        let intervalId;
        function moveSlide() {
          if (currentSlide < SLIDES - 1) {
            container.style.transform = `translateX(-${(currentSlide + 1) * 100}%)`;
            currentSlide++;
          } else {
            container.style.transform = 'translateX(0)';
            currentSlide = 0;
          }
          intervalId = setTimeout(moveSlide, TIMEOUT);
        }
        function startSlider() {
          intervalId = setTimeout(moveSlide, TIMEOUT);
        }
        function stopSlider() {
          clearTimeout(intervalId);
        }
        container.addEventListener('mouseover', stopSlider);
        container.addEventListener('mouseout', startSlider);
        container.addEventListener('mousedown', stopSlider);
        container.addEventListener('mouseup', startSlider);
        setTimeout(moveSlide, TIMEOUT);
      }
    }

    // Establecer las redes sociales
    // @param {HTMLElement} container - Contenedor de las redes sociales
    // @param {object} data - Datos de las redes sociales
    function setupSocial(container, data) {
      if (!container || !data) return;
      const socialItems = Object.keys(data).map(key => {
        return `<a href="${data[key]}" target="_blank" class="player-social-link player-social-link-${key}">
      ${svgIcons[key]}
    </a>`;
      });
      container.innerHTML = socialItems.join('');
    }
    let currentVideo;

    // Iniciar la aplicación
    function initApp() {
      if (window.stream_active) return;
      window.stream_active = true;
      const STREAMS = window?.streams || {};
      const SERVICE = STREAMS?.service || 'spotify';
      const MODULE_VIDEO_TOPS = STREAMS?.module_video_tops;
      const MODULE_NEWS = STREAMS?.module_news;
      const MODULE_TEAM = STREAMS?.module_team;
      const CAROUSEL_TIME = STREAMS?.carouselTiming || 5000;
      const VIDEO_POSTER = STREAMS?.video_poster;

      // --- NUEVA VARIABLE DE TOKEN ---
      // Este token se incrementa cada vez que se inicia una nueva actualización de estación.
      let currentUpdateToken = 0;
      initOutsideClick();
      function play(audio, newSource = null) {
        if (newSource) {
          audio.src = newSource;
        }

        // Visualizer
        if (!hasVisualizer) {
          visualizer(audio, visualizerContainer);
          hasVisualizer = true;
        }
        audio.load();
        audio.play();
        playButton.innerHTML = icons.pause;
        playButton.classList.add('is-active');
        if (currentVideo) {
          // Verificar si videojs esta reproduciendo,si es así, pausar
          if (!currentVideo.paused()) {
            localStorage.setItem('videoPlaying', true);
            currentVideo.pause();
          }
        }
      }
      function pause(audio) {
        audio.pause();
        playButton.innerHTML = icons.play;
        playButton.classList.remove('is-active');

        // Verificar si el video se pausó, si es así, reproducir
        if (localStorage.getItem('videoPlaying')) {
          currentVideo.play();
          localStorage.removeItem('videoPlaying');
        }
      }

      // Botón play/pause, al pausar detener el stream, al reproducir iniciar el stream de nuevo
      // playButton, play, pause son funciones exportadas que se usaran en otros archivos
      const range = document.querySelector('.player-volume');
      const rangeFill = document.querySelector('.player-range-fill');
      const rangeWrapper = document.querySelector('.player-range-wrapper');
      const rangeThumb = document.querySelector('.player-range-thumb');
      let currentVolume;
      try {
        currentVolume = localStorage.getItem('volume') || 100;
      } catch (error) {
        currentVolume = 100;
        console.log('Error al obtener el volumen del localStorage:', error);
      }
      let currentStation;
      let currentActiveStation;
      let currentSongData = {};
      let customSongIsPlaying = false;
      let currentCustomEl;

      // Rango recorrido
      function setRangeWidth(percent) {
        {
          rangeFill.style.height = `${percent}%`;
        }
      }

      // Posición del thumb
      function setThumbPosition(percent) {
        const compensatedWidth = rangeWrapper.offsetHeight - rangeThumb.offsetHeight;
        const thumbPosition = percent / 100 * compensatedWidth;
        {
          rangeThumb.style.bottom = `${thumbPosition}px`;
        }
      }

      // Actualiza el volumen al cambiar el rango
      function updateVolume(audio, value) {
        range.value = value;
        setRangeWidth(value);
        setThumbPosition(value);
        audio.volume = value / 100;
        if (localStorage === null || localStorage === undefined) return;
        localStorage.setItem('volume', value);
      }
      const playButton = document.querySelector('.player-button-play');
      const visualizerContainer = document.querySelector('.visualizer');
      let hasVisualizer = false;
      const audio = new Audio();
      audio.crossOrigin = 'anonymous';
      if (playButton !== null) {
        playButton.addEventListener('click', async () => {
          if (audio.paused) {
            play(audio);
          } else {
            pause(audio);
          }
        });
      }

      // Valor inicial
      if (range !== null) {
        updateVolume(audio, currentVolume);

        // Escucha el cambio del rango
        range.addEventListener('input', event => {
          updateVolume(audio, event.target.value);
        });

        // Escucha el movimiento del mouse
        rangeThumb.addEventListener('mousedown', () => {
          document.addEventListener('mousemove', handleThumbDrag);
        });
      }

      // Mueve el thumb y actualiza el volumen
      function handleThumbDrag(event) {
        const rangeRect = range.getBoundingClientRect();
        const click = event.clientY - rangeRect.top;
        let percent = click / range.offsetWidth * 100;
        percent = 100 - percent;
        percent = Math.max(0, Math.min(100, percent));
        const value = Math.round((range.max - range.min) * (percent / 100)) + parseInt(range.min);
        updateVolume(audio, value);
      }

      // Deja de escuchar el movimiento del mouse
      document.addEventListener('mouseup', () => {
        document.removeEventListener('mousemove', handleThumbDrag);
      });
      window.addEventListener('resize', () => {
        const currentPercent = range.value;
        setRangeWidth(currentPercent);
        setThumbPosition(currentPercent);
      });

      // ----------------------------------------------
      // Resto del código
      // ----------------------------------------------
      const appSuper = document.getElementById('app-super');
      const player = document.getElementById('super-player');
      const playerExpandButton = document.getElementById('player-expand');
      const hiddenPlayerButton = document.getElementById('player-down-up');
      const template = document.querySelector('.template');
      if (hiddenPlayerButton) {
        hiddenPlayerButton.addEventListener('click', () => {
          if (player.classList.contains('is-hidden')) {
            player.classList.remove('is-hidden');
            template.classList.remove('player-hidden');
          } else {
            player.classList.toggle('is-hidden');
            template.classList.toggle('player-hidden');
          }
        });
      }
      if (playerExpandButton) {
        playerExpandButton.addEventListener('click', () => {
          if (appSuper.classList.contains('is-expanded')) {
            document.body.style.overflow = 'auto';
            playerExpandButton.classList.remove('is-open');
            appSuper.classList.remove('is-expanded');
          } else {
            document.body.style.overflow = 'hidden';
            playerExpandButton.classList.add('is-open');
            appSuper.classList.add('is-expanded');
          }
        });
      }
      const songNow = document.querySelector('.song-now');
      const history = document.getElementById('historyp');
      const historyTemplate = `<div class="historyp-item flex items-center g-0.75">
  <div class="historyp-image flex-none">
    <img src="{{art}}" width="80" height="80">
  </div>
  <div class="historyp-body truncate-line">
    <span class="historyp-name color-title fw-500">{{song}}</span>
    <span class="historyp-time color-text" data-time-ago="{{timeAgo}}">{{artist}}</span>
  </div>
</div>`;
      let observer;
      const TIME_TO_REFRESH = window?.streams?.timeRefresh || 10000;
      function songProgress(duration, elapsed) {
        const progress = getPercentage(elapsed, duration);

        // Agregar la variable css con el progreso de la canción
        appSuper.style.setProperty('--song-progress', `${2 + progress}%`);

        // Actualizar el progreso de la canción
        const songElapsed = document.querySelector('.song-elapsed');
        const songDuration = document.querySelector('.song-duration');
        if (songElapsed) {
          songElapsed.textContent = convertSecondsToMinutes(elapsed);
        }
        if (songDuration) {
          songDuration.textContent = convertSecondsToMinutes(duration);
        }
      }
      function setMediaSession(data, current = null, token) {
        // Si en algún momento el token no coincide, se descarta esta actualización.
        if (token !== currentUpdateToken) return;
        const {
          title,
          artist,
          album,
          artwork
        } = data;
        if ('mediaSession' in navigator) {
          navigator.mediaSession.metadata = new MediaMetadata({
            title,
            artist,
            album,
            artwork: [{
              src: artwork || current.picture,
              sizes: '128x128',
              type: 'image/png'
            }, {
              src: artwork || current.picture,
              sizes: '256x256',
              type: 'image/png'
            }, {
              src: artwork || current.picture,
              sizes: '512x512',
              type: 'image/png'
            }]
          });
          navigator.mediaSession.setActionHandler('play', () => {
            play(audio);
          });
          navigator.mediaSession.setActionHandler('pause', () => {
            pause(audio);
          });
        }
      }
      function playerCustomPlay() {
        const playButton = document.querySelector('.player-custom-play');
        if (playButton) {
          playButton.addEventListener('click', event => {
            event.preventDefault();
            if (!audio.paused) {
              return;
            }
            play(audio);
          });
        }
      }
      function scrollText(container) {
        const item = container.querySelector('.song-title');
        if (item.scrollWidth > container.offsetWidth) {
          item.classList.add('is-scrolling');
          const scroll = item.scrollWidth - container.offsetWidth;
          const duration = 5000 + scroll * 100;
          item.setAttribute('style', `--text-scroll-duration: ${duration}ms; --text-scroll: -${scroll}px`);
          return;
        }
        item.classList.remove('is-scrolling');
        item.removeAttribute('style');
      }
      function normalizeCustomSong(data, baseUrl) {
        /* eslint-disable camelcase */
        const {
          pod_titulo,
          pod_desc,
          pod_url,
          pod_imagen
        } = data;
        const songData = {
          mp3: pod_url,
          title: pod_titulo,
          artist: pod_desc,
          artworkUrl100: `${baseUrl}${pod_imagen}`,
          artworkUrl500: `${baseUrl}${pod_imagen}`,
          artworkUrl1000: `${baseUrl}${pod_imagen}`
        };
        /* eslint-enable camelcase */

        return songData;
      }
      function updateSongProgress() {
        songProgress(audio.duration, audio.currentTime);
      }
      function playCustomSong(el, data) {
        // Volver a la estación original que estaba sonando
        if (customSongIsPlaying && currentCustomEl === el) {
          el.innerHTML = icons.play;
          audio.src = currentStation.stream_url;
          setCurrentSong(currentSongData);
          setMediaSession(currentSongData);
          play(audio);
          customSongIsPlaying = false;
          audio.removeEventListener('timeupdate', updateSongProgress);
          el.parentNode.classList.remove('is-playing');
          return;
        }
        const topsButtons = document.querySelectorAll('.nexo-tops-button');
        topsButtons.forEach(button => {
          button.innerHTML = icons.play;
          currentCustomEl?.parentNode.classList.remove('is-playing');
        });
        el.innerHTML = icons.stop;
        currentCustomEl = el;

        // agregar al padre del "el" la clase "is-playing"
        el.parentNode.classList.add('is-playing');
        setCurrentSong(data);
        setMediaSession(data);
        audio.src = data.mp3;
        audio.addEventListener('timeupdate', updateSongProgress);
        play(audio);
        customSongIsPlaying = true;
        /* eslint-enable camelcase */
      }
      function setCurrentSong(data, current, token) {
        // Si en algún momento el token no coincide, se descarta esta actualización.
        if (token !== currentUpdateToken) return;
        const {
          title,
          artist,
          artwork
        } = data;
        const content = songNow;
        const songName = document.querySelectorAll('.song-name');
        const songArtist = document.querySelectorAll('.song-artist');
        songName.forEach(item => {
          item.textContent = title;
        });
        songArtist.forEach(item => {
          item.textContent = artist;
        });
        const songTitle = content.querySelector('.player-title-fix');
        scrollText(songTitle);
        if (observer) {
          observer.disconnect();
        }
        observer = new ResizeObserver(() => {
          scrollText(songTitle);
        });
        observer.observe(songTitle);
        const pictureEl = content.querySelector('.player-artwork');
        const artworkUrl = artwork || current.picture || pixel;
        const playerLeft = document.querySelector('.player-left');
        playerLeft.style.removeProperty('--artwork');
        playerLeft.style.setProperty('--artwork', `url(${artworkUrl})`);
        if (pictureEl) {
          loadImage(artworkUrl).then(() => {
            // Verifica nuevamente el token antes de aplicar la transición
            if (token === currentUpdateToken) {
              slideUpImageTransition(pictureEl, artworkUrl);
            }
          }).catch(() => {
            console.error('Error al cargar arte de la canción');
          });
        }
        const coverEl = document.querySelector('.player-bg');
        // const coverHeaderEl = document.querySelector('.player-header-cover')
        const coverUrl = artwork || current.picture || pixel;
        if (coverEl) {
          const $img = document.createElement('img');
          $img.src = coverUrl;

          // eslint-disable-next-line no-undef
          const colorThief = new ColorThief();
          createTempImage($img.src).then(img => {
            setAccentColor(img, colorThief);
          });
          loadImage(coverUrl).then(() => {
            if (token === currentUpdateToken) {
              fadeImageTransition(coverEl, coverUrl);
            }
            // fadeImageTransition(coverHeaderEl, coverUrl)
          }).catch(() => {
            console.error('Error al cargar la portada de la canción');
          });
        }
      }
      function forcePlay(audio) {
        if (audio.paused) {
          play(audio);
        }
      }

      // Establecer las canciones que se han reproducido
      function setHistory(data, current, token) {
        // Si en algún momento el token no coincide, se descarta esta actualización.
        if (token !== currentUpdateToken) return;
        if (!history) return;
        history.innerHTML = historyTemplate.replace('{{art}}', pixel).replace('{{song}}', 'Cargando historial...').replace('{{artist}}', 'Artista').replace('{{timeAgo}}', 'time-none');
        if (!data) return;
        const promises = data.map(async item => {
          const {
            song: {
              artist,
              title,
              playedAt,
              elapsed
            }
          } = item;
          const dataFrom = await getDataFromSearchApi(`${artist} - ${title}`, SERVICE);
          if (dataFrom.artwork.includes('https://nexoapi.lat/') || dataFrom.artwork === null || title.trim() === '') {
            dataFrom.artwork = current.picture || current.cover;
            dataFrom.title = current.name;
            dataFrom.artist = current.description;
          } else {
            dataFrom.title = title;
            dataFrom.artist = artist;
          }
          const newHistoryTemplate = historyTemplate.replace('{{art}}', dataFrom.artwork).replace('{{song}}', dataFrom.title).replace('{{artist}}', dataFrom.artist).replace('{{timeAgo}}', 'time-none');
          if (elapsed) {
            return newHistoryTemplate.replace('time-none', 'Ahora');
          }
          if (playedAt) {
            return newHistoryTemplate.replace('time-none', timeAgo(unixToDate(playedAt)));
          }
          return newHistoryTemplate;
        });
        Promise.all(promises).then(itemsHTML => {
          const $fragment = document.createDocumentFragment();
          itemsHTML.forEach(itemHTML => {
            $fragment.appendChild(createElementFromHTML(itemHTML));
          });
          history.innerHTML = '';
          history.appendChild($fragment);
        }).catch(error => {
          console.error('Error:', error);
        });
      }

      // Establecer las canciones que se han reproducido
      async function setPageHistory(data, current, token) {
        // Si en algún momento el token no coincide, se descarta esta actualización.
        if (token !== currentUpdateToken) return;
        const historyContainer = document.querySelector('.last-played');
        if (!historyContainer || !data) return;
        data = data.slice(0, 8);

        // Array para almacenar los datos de las canciones
        const historyCurrentData = [];
        const itemsHTML = await Promise.all(data.map(async ({
          song: {
            artist,
            title
          }
        }, index) => {
          const dataFrom = await getDataFromSearchApi(`${artist} - ${title}`, SERVICE);
          if (dataFrom.artwork.includes('https://nexoapi.lat/') || dataFrom.artwork === null || title.trim() === '') {
            dataFrom.artwork = current.picture || current.cover;
            dataFrom.title = current.name;
            dataFrom.artist = current.description;
          } else {
            dataFrom.title = title;
            dataFrom.artist = artist;
          }

          // Almacenar los datos en el índice correcto
          historyCurrentData[index] = dataFrom;
          return `
        <div class="history" data-index="${index}">
          <div class="history-wrapper">
            <img class="history-art" src="${dataFrom.artwork}" alt="${dataFrom.title}">
            <div class="history-meta fs-7">
              <span class="history-title fw-700 uppercase">${dataFrom.title}</span>
              <span class="history-artist">${dataFrom.artist}</span>
            </div>
          </div>
          <a class="history-url history-deezer" href="${dataFrom.stream || '#not-found'}" target="_blank">
            ${svgIcons.deezer}
          </a>
          <button class="history-button"></button>
        </div>`;
        }));
        if (token === currentUpdateToken) {
          historyContainer.innerHTML = itemsHTML.join('');
          const historyItemsElements = historyContainer.querySelectorAll('.history');
          historyItemsElements.forEach(item => {
            const button = item.querySelector('.history-button');
            const index = item.getAttribute('data-index');
            button.addEventListener('click', event => {
              // Abrir modal con la canción seleccionada
              event.preventDefault();
              openModal('song', {
                data: historyCurrentData[index]
              });
            });
          });
        }
      }
      function loadTops(url) {
        window.mixxFunctions.dataTops = url;
        const dataTopsEl = document.getElementById('player-tops');
        if (!dataTopsEl) return;
        fetch(url).then(response => response.json()).then(data => {
          const template = `<div class="nexo-tops-item">
          <div class="nexo-tops-head">
            <div class="nexo-tops-image-container">
              <span>{{number}}</span>
              <img src="{{img}}" alt="imagen" class="nexo-tops-image" width="100">
            </div>
            <div class="nexo-tops-data">
              <div class="nexo-tops-title">{{title}}</div>
              <span class="nexo-tops-artist">{{artist}}</span>
            </div>
          </div>
          <div class="nexo-tops-footer">
            <div class="nexo-tops-footer-content">
              <button class="nexo-tops-button">${icons.play}</button>
              <div class="nexo-tops-meta">
                <div class="nexo-tops-meta-title">{{title}}</div>
                <div class="nexo-tops-meta-artist">{{artist}}</div>
              </div>
            </div>
          </div>
        </div>`;
          const $fragment = document.createDocumentFragment();
          data.forEach((item, index) => {
            const imgUrl = `${baseUrl}${item.pod_imagen}`;
            const newTemplate = template.replace('{{number}}', index + 1).replace('{{img}}', imgUrl).replace(/{{title}}/g, item.pod_titulo).replace(/{{artist}}/g, item.pod_desc);
            const $item = createElementFromHTML(newTemplate);
            const $button = $item.querySelector('.nexo-tops-button');
            const songData = normalizeCustomSong(item, baseUrl);
            $button.addEventListener('click', () => {
              playCustomSong($button, songData);
            });
            $fragment.appendChild($item);
          });
          dataTopsEl.innerHTML = '';
          dataTopsEl.appendChild($fragment);
          const $elements = document.querySelectorAll('.nexo-tops-head');
          let currentElement;
          $elements.forEach(element => {
            element.addEventListener('click', () => {
              if (currentElement === element) {
                element.classList.remove('is-active');
                currentElement = null;
                return;
              }
              if (currentElement) {
                currentElement.classList.remove('is-active');
              }
              element.classList.toggle('is-active');
              currentElement = element;
            });
          });
        }).catch(error => {
          console.error('Error:', error);
        });
      }
      function loadVideoTops(url) {
        window.mixxFunctions.dataTops = url;
        const dataTopsEl = document.getElementById('player-video-tops');
        if (!dataTopsEl) return;
        const listContainer = dataTopsEl.querySelector('.nexo-vid-tops-list');
        const videoContainer = dataTopsEl.querySelector('.nexo-vid-tops-video');
        if (!dataTopsEl) return;
        let currentElement;
        let currentVideo; // Declaramos aquí para que sea accesible en todo el contexto.

        function setVideo(url, thumbnail, play = false) {
          const $video = document.createElement('video');
          if (currentVideo && play) {
            currentVideo.pause();
            currentVideo.dispose(); // Eliminamos el video anterior
          }
          $video.id = 'player';
          $video.poster = thumbnail;
          $video.classList.add('video-js', 'video-js-stream');
          videoContainer.innerHTML = '';
          videoContainer.appendChild($video);

          // eslint-disable-next-line no-undef
          const video = videojs($video, {
            sources: [{
              src: url
            }],
            controls: true
          });
          currentVideo = video;
          video.ready(() => {
            if (play) {
              setTimeout(() => {
                video.play()?.then(() => {
                  console.log('Reproduciendo video');
                  if (appPlayer) pause(audio);
                }).catch(() => {
                  console.error('Error al reproducir video');
                });
              }, 500);
            }
            video.on('play', () => {
              if (appPlayer) pause(audio);
            });
            video.on('pause', () => {
              if (appPlayer) forcePlay(audio);
            });
          });
        }
        fetch(url).then(response => response.json()).then(data => {
          const template = `<div class="nexo-vid-tops-item">
          <button class="nexo-vid-tops">
            <div class="nexo-vid-tops-image-container">
              <span>{{number}}</span>
              <img src="{{img}}" alt="imagen" class="nexo-vid-tops-image" width="140">
            </div>
            <div class="nexo-vid-tops-data">
              <div class="nexo-vid-tops-title">{{title}}</div>
              <span class="nexo-vid-tops-artist">{{description}}</span>
            </div>
          </button>
        </div>`;
          const $fragment = document.createDocumentFragment();
          data.forEach((item, index) => {
            const imgUrl = `${baseUrl}${item.photo}`;
            const vidUrl = `${baseUrl}${item.video}`;
            if (index === 0) {
              setVideo(vidUrl, imgUrl);
            }
            const newTemplate = template.replace('{{number}}', index + 1).replace('{{img}}', imgUrl).replace(/{{title}}/g, item.nombre).replace(/{{description}}/g, item.descripcion);
            const $item = createElementFromHTML(newTemplate);
            $fragment.appendChild($item);
          });
          listContainer.innerHTML = '';
          listContainer.appendChild($fragment);
          const $elements = document.querySelectorAll('.nexo-vid-tops');
          $elements.forEach((element, index) => {
            if (index === 0) {
              element.classList.add('is-active');
              currentElement = element;
            }
            element.addEventListener('click', () => {
              if (currentElement === element) {
                return;
              }
              if (currentElement) {
                currentElement.classList.remove('is-active');
              }
              element.classList.add('is-active');
              currentElement = element;
              const vidUrl = `${baseUrl}${data[index].video}`;
              const imgUrl = `${baseUrl}${data[index].photo}`;
              setVideo(vidUrl, imgUrl, true); // Llamamos a setVideo con "true" para que reproduzca automáticamente
            });
          });
        }).catch(error => {
          console.error('Error:', error);
        });
      }
      const videoModule = document.getElementById('player-video-module');
      function initVideoPlayer(videoUrl) {
        if (currentVideo) {
          currentVideo?.dispose();
        }
        const $video = document.createElement('video');
        $video.id = 'player';
        $video.classList.add('video-js', 'video-js-stream');
        videoModule.innerHTML = '';
        videoModule.appendChild($video);
        const videoId = document.getElementById('player');

        // eslint-disable-next-line no-undef
        currentVideo = videojs(videoId, {
          sources: [{
            src: videoUrl,
            type: 'application/x-mpegURL'
          }],
          poster: VIDEO_POSTER,
          controls: true,
          autoplay: true
        });
      }
      function setArticles(data, baseUrl) {
        const newsContainer = document.getElementById('news');
        if (!newsContainer || !data) return;
        data = data.slice(0, 8);
        for (const item of data) {
          const {
            titulo,
            photo,
            contenido
          } = item;
          const time = new Date(item.created_at);
          const createdAt = timeAgo(time);
          const article = document.createElement('article');
          article.classList.add('news-item', 'swiper-slide');
          article.innerHTML = `
        <div class="news-image">
          <img src="${baseUrl}${photo}" alt="${titulo}">
        </div>
        <div class="news-content">
          <h3 class="news-title">${titulo}</h3>
          <time class="news-date">${createdAt}</time>
        </div>
      `;
          article.addEventListener('click', () => {
            openModal('article', {
              data: {
                titulo,
                contenido,
                imagen: `${baseUrl}${photo}`,
                time: createdAt
              }
            });
          });
          newsContainer.appendChild(article);
        }

        // eslint-disable-next-line no-undef, no-unused-vars
        new Swiper('.swiper-news', {
          // Enabled autoplay mode
          autoplay: {
            delay: CAROUSEL_TIME,
            disableOnInteraction: false
          },
          // If we need navigation
          navigation: {
            nextEl: '.swiper-next-news',
            prevEl: '.swiper-prev-news'
          },
          // Responsive breakpoints
          breakpoints: {
            480: {
              slidesPerView: 2,
              spaceBetween: 12
            },
            640: {
              slidesPerView: 3,
              spaceBetween: 12
            },
            1024: {
              slidesPerView: 4,
              spaceBetween: 12
            }
          }
        });
      }
      function setTeam(data, baseUrl) {
        const teamContainer = document.getElementById('team');
        if (!teamContainer || !data) return;
        data = data.slice(0, 8);
        for (const item of data) {
          const equipoNombre = item.equipo_nombre;
          const equipoFoto = item.equipo_foto;
          const equipoDescripcion = item.equipo_descripcion;
          const article = document.createElement('article');
          article.classList.add('team-item', 'swiper-slide');
          article.innerHTML = `
        <div class="team-image">
          <img src="${baseUrl}${equipoFoto}" alt="${equipoNombre}">
        </div>
        <div class="team-content">
          <h3 class="team-title">${equipoNombre}</h3>
          <p class="team-description">${equipoDescripcion}</p>
        </div>
      `;
          teamContainer.appendChild(article);
        }

        // eslint-disable-next-line no-undef, no-unused-vars
        new Swiper('.swiper-team', {
          // Enabled autoplay mode
          autoplay: {
            delay: CAROUSEL_TIME,
            disableOnInteraction: false
          },
          // If we need navigation
          navigation: {
            nextEl: '.swiper-next-team',
            prevEl: '.swiper-prev-team'
          },
          // Responsive breakpoints
          breakpoints: {
            480: {
              slidesPerView: 2,
              spaceBetween: 12
            },
            640: {
              slidesPerView: 3,
              spaceBetween: 12
            },
            1024: {
              slidesPerView: 4,
              spaceBetween: 12
            }
          }
        });
      }
      window.mixxFunctions = {
        loadTops,
        loadVideoTops
      };

      // --------------------------------------------------------------------------------------------//
      // INICIAR LA APLICACIÓN ----------------------------------------------------------------------//
      // --------------------------------------------------------------------------------------------//
      let currentSongPlaying;
      let timeoutId;
      let timeoutIdProgram;
      let loadStations = [];
      const json = window.streams || {};
      const appPlayer = document.getElementById('app-player');
      const playerWrapper = document.querySelector('.player-wrapper');
      const hasLyrics = json.lyrics || false;
      const hasProgram = json.program || false;
      const hasExpand = json.expand || false;
      const hasShare = json.share_buttons || false;
      const baseUrl = json.base_url || '';
      const urlServer = json.url_server || 'api/';
      const idUser = json.id_user || '';
      const isMultiStream = json.multi_stream || false;
      const radioPath = isMultiStream ? 'multiradio-app/' : 'radio-app/';
      const dataUrl = `${baseUrl}${urlServer}${radioPath}${idUser}`;
      const dataSocialUrl = `${baseUrl}${urlServer}radio-app/${idUser}`;
      const dataAppsUrl = `${baseUrl}${urlServer}radio-app/${idUser}`;
      const dataProgramUrl = `${baseUrl}${urlServer}programa-app/${idUser}`;
      const dataBannersUrl = `${baseUrl}${urlServer}slide-app/${idUser}`;
      const dataPromoUrl = `${baseUrl}${urlServer}promocion-app/${idUser}`;
      const dataGetPromoUrl = `${baseUrl}${urlServer}promotional-contact`;
      const dataTops = `${baseUrl}${urlServer}podcast-app/${idUser}`;
      const dataTopsVid = `${baseUrl}${urlServer}top-app/videos/${idUser}`;
      const dataNoticiasUrl = `${baseUrl}${urlServer}noticia-app/${idUser}`;
      const dataVideoUrl = `${baseUrl}${urlServer}video-app/${idUser}`;
      const dataTeamUrl = `${baseUrl}${urlServer}equipo-app/${idUser}`;
      const buttonLyrics = document.querySelector('.player-button-lyrics');
      const buttonProgram = document.querySelector('.player-button-program');
      const buttonExpand = document.querySelector('.player-button-expand');
      const buttonPrev = document.querySelector('.player-button-prev');
      const buttonNext = document.querySelector('.player-button-next');
      const buttonShare = document.querySelector('.player-button-share');
      const stationsButton = document.querySelector('.station-prm');
      const playerSocial = document.querySelector('.player-social');
      function enableButtons() {
        if (hasLyrics) {
          buttonLyrics.classList.add('is-loaded');
        }
        if (hasProgram) {
          buttonProgram.classList.add('is-loaded');
        }
        if (hasExpand) {
          buttonExpand.classList.add('is-loaded');
        }
        if (isMultiStream) {
          buttonPrev.classList.add('is-loaded');
          buttonNext.classList.add('is-loaded');
        }
        if (hasShare) {
          buttonShare.classList.add('is-loaded');
        }
        playerWrapper.addEventListener('touchmove', function (e) {
          e.preventDefault();
        }, {
          passive: false
        });
      }
      window.addEventListener('DOMContentLoaded', () => {
        loadVideoTops(dataTopsVid);
      });
      if (!appPlayer) return;
      enableButtons();
      fetch(dataUrl).then(response => response.json()).then(data => {
        if (isMultiStream) {
          const stations = data.map(item => {
            return {
              name: item.multiradio_nombre,
              description: item.multiradio_descripcion,
              stream_url: item.multiradio_url,
              api: item.multiradio_api,
              picture: `${baseUrl}${item.multiradio_imagen}`,
              background: pixel,
              tv_url: item.multiradio_tvstream,
              id: item.id
            };
          });
          loadStations = stations;
          setupMiniSchedule(appPlayer, stations[0]);
          setupStream(appPlayer, stations[0]);
          audio.src = stations[0].stream_url;
          if (stations[0].tv_url) {
            initVideoPlayer(stations[0].tv_url);
          } else {
            currentVideo?.dispose();
            currentVideo = null;
            videoModule.innerHTML = '<div class="novids">Sin video</div>';
          }
        } else {
          const station = {
            name: data.radio_nombre,
            stream_url: data.radio_url,
            tv_url: data.radio_video,
            picture: `${baseUrl}${data.radio_fondo}`,
            background: `${baseUrl}${data.radio_splash}`,
            api: data.radio_menu_nicio
          };
          currentStation = station;
          setupMiniSchedule(appPlayer, station);
          setupStream(appPlayer, station);
          audio.src = station.stream_url;
          if (station.tv_url) {
            initVideoPlayer(station.tv_url);
          } else {
            currentVideo?.dispose();
            currentVideo = null;
            videoModule.innerHTML = '<div class="novids">Sin video</div>';
          }
        }
        function init(current, change = false) {
          // Incrementar el token de actualización
          const token = ++currentUpdateToken;
          currentStation = current;
          if (timeoutId) clearTimeout(timeoutId);
          const fetchUrl = `api/met?url=${current.stream_url}`;
          fetch(fetchUrl).then(response => response.json()).then(async data => {
            // Si en el intento se cambió la estación, se descarta esta respuesta
            if (token !== currentUpdateToken) return;
            if (data.now_playing && !customSongIsPlaying) {
              songProgress(data.now_playing.duration, data.now_playing.elapsed);
            }
            const title = data.title || data.song;
            const query = title.trim() !== '' ? `${data.artist} - ${title}` : `${current.name} - ${current.descripcion}`;
            if (currentSongPlaying !== query) {
              currentSongPlaying = query;
              const songData = await getDataFromSearchApi(query, SERVICE);
              if (songData.artwork.includes('https://nexoapi.lat/') || songData.artwork === null || title.trim() === '') {
                songData.artwork = current.picture || current.cover;
                songData.title = current.name;
                songData.artist = current.description;
              } else {
                songData.title = title;
                songData.artist = data.artist;
              }
              currentSongData = songData;
              setCurrentSong(songData, current, token);
              setMediaSession(songData, current, token);
              setPageHistory(data.song_history, current, token);

              // Agregar la cancion actual al inicio de la lista de canciones
              if (data.song_history && data.song_history.length > 0) {
                data.song_history.unshift({
                  song: {
                    artist: songData.artist,
                    title: songData.title,
                    elapsed: data.now_playing.elapsed,
                    duration: data.now_playing.duration
                  }
                });
              }
              setHistory(data.song_history, current, token);
              const lyricsData = await getLyrics(songData.artist, songData.title);
              const lyricsButton = document.querySelector('.player-button-lyrics');
              if (lyricsButton) {
                lyricsButton.onclick = () => {
                  openModal('lyrics', {
                    lyrics: lyricsData
                  });
                };
              }
              if (buttonShare) {
                buttonShare.onclick = () => {
                  openModal('share', {
                    data: songData
                  });
                };
              }
            }
          }).catch(error => console.log(error));
          if (hasProgram) {
            fetch(dataProgramUrl).then(response => response.json()).then(data => {
              const container = document.getElementById('player-programs');
              setupMiniSchedule(container, data, baseUrl);
            }).catch(error => {
              console.error('Error:', error);
            });
          }
          timeoutId = setTimeout(() => {
            init(current);
          }, TIME_TO_REFRESH);
        }

        // DATOS DE LA PROGRAMACIÓN
        function initProgram() {
          if (timeoutIdProgram) clearTimeout(timeoutIdProgram);
          fetch(dataProgramUrl).then(response => response.json()).then(data => {
            setupSchedule(data, baseUrl);
          }).catch(error => {
            console.error('Error:', error);
          });
          // Refrescar cada 1 minuto
          timeoutIdProgram = setTimeout(() => {
            initProgram();
          }, 60000);
        }
        initProgram();

        // Botones Apps
        fetch(dataAppsUrl).then(response => response.json()).then(data => {
          setupApps(getApps(data)); // Llama a la función setupApps con el resultado de getApps
        }).catch(error => {
          console.error('Error:', error);
        });
        function setupApps(apps) {
          // Verifica si hay URL para Android y actualiza el botón
          if (apps.android) {
            const androidButton = document.getElementById('androidButton');
            androidButton.href = apps.android; // Asigna la URL para Android
            androidButton.style.display = 'inline-block'; // Asegura que se muestre el botón
          } else {
            const androidButton = document.getElementById('androidButton');
            androidButton.style.display = 'none'; // Oculta el botón si no hay URL
          }

          // Verifica si hay URL para iOS y actualiza el botón
          if (apps.ios) {
            const iosButton = document.getElementById('iosButton');
            iosButton.href = apps.ios; // Asigna la URL para iOS
            iosButton.style.display = 'inline-block'; // Asegura que se muestre el botón
          } else {
            const iosButton = document.getElementById('iosButton');
            iosButton.style.display = 'none'; // Oculta el botón si no hay URL
          }
        }

        // DATOS DE LOS BANNERS
        fetch(dataBannersUrl).then(response => response.json()).then(data => {
          const container = document.querySelector('.slider-wrapper');
          setupSlider(container, data, baseUrl);
        }).catch(error => {
          console.error('Error:', error);
        });
        function initSocial() {
          if (!playerSocial) return;

          // Botones sociales
          fetch(dataSocialUrl).then(response => response.json()).then(data => {
            setupSocial(playerSocial, getSocials(data));
          }).catch(error => {
            console.error('Error:', error);
          });
        }
        initSocial();
        const promoContainer = document.getElementById('player-prm-send');
        if (promoContainer) {
          // DATOS DE LAS PROMOCIONES
          fetch(dataPromoUrl).then(response => response.json()).then(data => {
            if (data.length === 0) return;
            const button = document.querySelector('.send-prm');
            button.classList.remove('none');
            data.forEach(item => {
              const img = document.createElement('img');
              img.src = baseUrl + item.photo;
              img.classList.add('prm-image');
              img.addEventListener('click', () => {
                openModal('form', {
                  formAction: dataGetPromoUrl,
                  formId: item.promocion_id
                });
              });
              promoContainer.appendChild(img);
            });
          }).catch(error => {
            console.error('Error:', error);
          });
        }

        // DATOS DE LAS NOTICIAS
        function initNoticias() {
          const newsContainer = document.querySelector('.main-news');
          if (!newsContainer) return;
          fetch(dataNoticiasUrl).then(response => response.json()).then(data => {
            if (data.length === 0) return;
            setArticles(data, baseUrl);
            newsContainer.classList.remove('none');
          }).catch(error => {
            console.error('Error:', error);
          });
        }
        if (MODULE_NEWS) initNoticias();

        // DATOS DE LAS NOTICIAS
        function initTeam() {
          const teamContainer = document.querySelector('.main-team');
          if (!teamContainer) return;
          fetch(dataTeamUrl).then(response => response.json()).then(data => {
            if (data.length === 0) return;
            setTeam(data, baseUrl);
            teamContainer.classList.remove('none');
          }).catch(error => {
            console.error('Error:', error);
          });
        }
        if (MODULE_TEAM) initTeam();
        function initVideoApp() {
          const container = document.querySelector('.main-ytlist');
          if (!container) return;
          const CONTAINER_LIST = container.querySelector('.ytlistmini-list');
          if (!CONTAINER_LIST) return;
          fetch(dataVideoUrl).then(response => response.json()).then(data => {
            container.classList.remove('none');
            fetchVideoList(data, CONTAINER_LIST);
          }).catch(error => {
            console.error('Error:', error);
          });
        }
        if (MODULE_VIDEO_TOPS) initVideoApp();
        if (loadStations.length > 1) {
          stationsButton.classList.remove('none');
          const $stations = document.getElementById('player-prm');
          if (!$stations) return;
          $stations.classList.add('is-visible');
          loadStations.forEach((station, index) => {
            const img = document.createElement('img');
            img.classList.add('prm-image');
            img.src = station.picture;
            if (index === 0) {
              img.classList.add('is-current');
              currentActiveStation = img;
            }
            img.addEventListener('click', event => {
              if (currentStation === station) return;
              if (currentActiveStation) {
                currentActiveStation.classList.remove('is-current');
              }
              currentActiveStation = img;
              currentActiveStation.classList.add('is-current');
              init(station, true);
              setupStream(appPlayer, station);
              audio.src = station.stream_url;
              // play(audio)

              if (station.tv_url) {
                initVideoPlayer(station.tv_url);
              } else {
                currentVideo?.dispose();
                currentVideo = null;
                videoModule.innerHTML = '<div class="novids">Sin video</div>';
              }
              currentStation = station;
            });
            $stations.appendChild(img);
          });
          const $nextButton = buttonNext;
          const $prevButton = buttonPrev;
          $nextButton.addEventListener('click', () => {
            const $nextStation = currentActiveStation.nextElementSibling;
            if ($nextStation) {
              $nextStation.click();
            }
            if (!$nextStation && currentActiveStation) {
              const $firstStation = document.querySelector('.prm-image');
              if ($firstStation) {
                $firstStation.click();
              }
            }
          });
          $prevButton.addEventListener('click', () => {
            const $prevStation = currentActiveStation.previousElementSibling;
            if ($prevStation) {
              $prevStation.click();
            }
            if (!$prevStation && currentActiveStation) {
              const $lastStation = document.querySelector('.prm-image:last-child');
              if ($lastStation) {
                $lastStation.click();
              }
            }
          });
        }
        if (isMultiStream) {
          init(loadStations[0], true);
        } else {
          init(currentStation);
        }
      }).catch(error => {
        console.error('Error:', error);
      });
      window.addEventListener('DOMContentLoaded', () => {
        playerCustomPlay();
        loadTops(dataTops);
      });
      function setDeviceInfo() {
        const icons = {
          windows: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M11.5 3v8.5H3V3h8.5Zm0 18H3v-8.5h8.5V21Zm1-18H21v8.5h-8.5V3Zm8.5 9.5V21h-8.5v-8.5H21Z"/></svg>',
          android: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M6.4 4a9 9 0 0 1 11.2 0l1.5-1.5L20.5 4 19 5.4a9 9 0 0 1 2 5.6v1H3v-1c0-2.1.7-4 2-5.6L3.5 3.9 5 2.5 6.4 4ZM3 14h18v7c0 .6-.4 1-1 1H4a1 1 0 0 1-1-1v-7Zm6-5a1 1 0 1 0 0-2 1 1 0 0 0 0 2Zm6 0a1 1 0 1 0 0-2 1 1 0 0 0 0 2Z"/></svg>',
          apple: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M11.7 7.2c-.9 0-2.3-1-3.7-1A5.6 5.6 0 0 0 3.4 9c-2 3.4-.5 8.5 1.4 11.2 1 1.4 2 2.9 3.5 2.8 1.4 0 2-.9 3.7-.9 1.7 0 2.1 1 3.6.9 1.5 0 2.5-1.4 3.4-2.7 1.1-1.6 1.5-3.1 1.6-3.2a4.9 4.9 0 0 1-.6-8.7 5.2 5.2 0 0 0-4-2.2c-2-.1-3.5 1-4.3 1Zm3-2.8c.9-1 1.4-2.3 1.2-3.6a5 5 0 0 0-3.2 1.7c-.7.9-1.4 2.2-1.2 3.5 1.2 0 2.5-.7 3.3-1.6Z"/></svg>'
        };
        if (navigator.userAgent.match(/Android/i)) {
          return `${icons.android} Instalar en Android`;
        }
        if (navigator.userAgent.match(/iPhone|iPad|iPod/i)) {
          return `${icons.apple} Instalar en iOS`;
        }
        return `${icons.windows} Instalar en Windows`;
      }

      // Función para instalar PWA en el navegador
      function installPWA() {
        const container = document.getElementById('install-container');
        const close = document.getElementById('close-install');
        const install = document.getElementById('install-pwa');
        const checkIosDevice = () => {
          const ua = navigator.userAgent;
          const vendor = navigator.vendor;
          const isSafari = /Safari/.test(ua) && /Apple Computer/.test(vendor) && !/CriOS/.test(ua); // Excluir Chrome

          return isSafari;
        };
        if (checkIosDevice()) return;
        if (!container) return;
        if (!install) return;
        close.addEventListener('click', () => {
          container.classList.remove('is-active');
        });
        install.innerHTML = setDeviceInfo();
        const isMobileDevice = () => {
          const mobileDeviceRegex = /Android|Windows Phone|Windows|iPhone|iPad|iPod|iOS|Macintosh/i;
          return mobileDeviceRegex.test(navigator.userAgent);
        };
        const isFirefox = () => {
          return navigator.userAgent.includes('Firefox');
        };
        let installPrompt = null;
        window.addEventListener('beforeinstallprompt', event => {
          event.preventDefault();
          installPrompt = event;
          if (isMobileDevice() && !isFirefox()) {
            container.classList.add('is-active');
          }
        });
        install.addEventListener('click', async () => {
          if (!installPrompt) {
            console.warn('No se puede instalar la PWA');
            return;
          }
          const result = await installPrompt.prompt();
          if (result.outcome === 'accepted') {
            console.warn('PWA instalada');
            container.classList.remove('is-active');
          } else {
            console.warn('PWA no instalada');
          }
        });
      }
      installPWA();
    }
    initApp();
    if (typeof jQuery !== 'undefined') {
      // eslint-disable-next-line no-undef
      jQuery(document).on('ajaxComplete', function () {
        window.mixxFunctions.loadTops(window.mixxFunctions.dataTops);
        window.mixxFunctions.loadVideoTops(window.mixxFunctions.dataTopsVid);
      });
    }
  }

  function smartplayer() {
    const playerTime = document.querySelector('.player-time');
    const playerDay = playerTime.querySelector('.player-day strong');
    const playerHour = playerTime.querySelector('.player-hour span');

    // Mostrar dia en español
    const getDay = date => {
      const options = {
        weekday: 'long'
      };
      return new Intl.DateTimeFormat('es-ES', options).format(date);
    };

    // Obtener hora y minutos
    const getHoursAndMinutes = date => {
      return `${date.getHours()}:${date.getMinutes()}`;
    };

    // Determinar si es AM o PM
    const getAmPm = date => {
      return date.getHours() >= 12 ? 'PM' : 'AM';
    };

    // Mostrar hora actual en formato de 12 horas
    const getTime12 = date => {
      return `${getHoursAndMinutes(date)} ${getAmPm(date)}`;
    };

    // Iniciar todo
    function initClock() {
      const currentDate = new Date();
      playerDay.textContent = getDay(currentDate);
      playerHour.textContent = getTime12(currentDate);
      setTimeout(() => {
        initClock();
      }, 1000);
    }
    initClock();
    const STREAMS = window?.streams || {};
    const config = {
      timeRefresh: STREAMS.timeRefresh || 10000,
      programRefresh: STREAMS.programRefresh || 60000,
      baseUrl: STREAMS.base_url || 'https://pro.latinohitstv.com/panelradiotv/server/public/',
      idUser: STREAMS.id_user || 2,
      service: STREAMS.service || 'deezer',
      marqueeSpeed: STREAMS.marqueeSpeed || 12,
      marqueeGap: STREAMS.marqueeGap || '2vw'
    };

    // Obtener los datos de la búsqueda
    // @param {string} query - Consulta de búsqueda
    // @param {string} service - Servicio de búsqueda
    // @returns {Promise<object>} - Datos de la búsqueda
    const cache = {};
    const getDataFromSearchApi = async (query, service) => {
      if (cache[query]) {
        return cache[query];
      }
      const streamUrl = `api/search.php?query=${encodeURIComponent(query)}&service=${service}`;
      const response = await fetch(streamUrl);
      const data = await response.json();

      // Si no responde
      if (!data.results) {
        return {};
      }
      const results = data.results;
      cache[query] = results;
      return results;
    };

    // Obtener datos del stream
    // @param {string} streamUrl - URL del stream
    // @returns {Promise<object>} - Datos del stream
    async function getStreamData(streamUrl) {
      if (!streamUrl) {
        return {};
      }
      const jsonUri = `api/met?url=${encodeURIComponent(streamUrl)}`;
      try {
        const response = await fetch(jsonUri);
        return response.json();
      } catch (error) {
        console.error('Error al obtener los datos del stream:', error);
        return {};
      }
    }
    const normalizeData = data => {
      return {
        name: data.radio_nombre,
        description: data.radio_descripcion || 'SONANDO EN VIVO',
        image: `${config.baseUrl}${data.radio_fondo}`,
        cover: `${config.baseUrl}${data.radio_splash}`,
        stream: data.radio_url,
        video: data.radio_video,
        whatsapp: data.radio_whatsapp,
        facebook: data.radio_facebook_url,
        web: data.radio_web,
        youtube: data.radio_youtube,
        instagram: data.radio_instagram
      };
    };
    function createQuery(data) {
      const title = data.title || data.song;
      return title.trim() !== '' ? `${data.artist} - ${title}` : `${data.name} - ${data.description}`;
    }
    const setDefaultData = ({
      data,
      song,
      stream
    }) => {
      if (data.title.trim() === '') {
        data.title = stream.name;
        data.artist = stream.description;
        data.artist_artwork = stream.cover;
        data.artwork = {
          small: stream.cover,
          medium: stream.cover,
          large: stream.cover,
          xl: stream.cover
        };
      } else {
        data.title = song.title || song.song;
        data.artist = song.artist;
      }
      return data;
    };

    // @param {HTMLElement} item - Elemento que contiene el texto
    function scrollText(item) {
      if (!item) return;
      const parent = item.parentElement;
      if (!parent) return;
      const parentWidth = parent.offsetWidth;
      const itemWidth = item.scrollWidth;
      const marqueeGap = config.marqueeGap;

      // Verificar si ya se está aplicando el efecto de scroll para evitar duplicaciones
      const isScrolling = item.classList.contains('is-infinite-scrolling');
      if (itemWidth > parentWidth) {
        if (!isScrolling) {
          // Guardar el contenido original antes de duplicarlo
          item.dataset.originalText = item.innerText;
          item.innerHTML = `<span>${item.dataset.originalText}</span>   <span>${item.dataset.originalText}</span>`;
          item.classList.add('is-infinite-scrolling');
        }
        const scroll = item.scrollWidth - parentWidth;
        const duration = 5000 + scroll * config.marqueeSpeed;
        {
          item.style.setProperty('--marquee-gap', `${marqueeGap === 'auto' ? parentWidth + 'px' : marqueeGap}`);
        }
        item.style.setProperty('--text-scroll-duration', `${duration}ms`);
      } else {
        removeTextScroll(item);
      }
    }

    // Restaurar el texto original y quitar animación
    function removeTextScroll(item) {
      if (!item) return;
      if (item.classList.contains('is-infinite-scrolling')) {
        item.innerHTML = item.dataset.originalText || item.innerText;
        item.removeAttribute('data-original-text');
        item.classList.remove('is-infinite-scrolling', 'is-backwards');
      }
    }

    // Observar cambios en el tamaño del contenedor
    function observeTextScroll(item) {
      if (!item || !item.parentElement) return;
      removeTextScroll(item);
      const resizeObserver = new ResizeObserver(() => scrollText(item));
      resizeObserver.observe(item.parentElement);
      item._resizeObserver = resizeObserver;
      scrollText(item);
    }

    // Devolver una promesa para saber si la imagen se ha cargado correctamente
    // @param {string} src - URL de la imagen
    // @returns {Promise} - Promesa que se resuelve si la imagen se carga correctamente
    function loadImage(src) {
      return new Promise((resolve, reject) => {
        const img = new Image();
        img.onload = () => resolve(img);
        img.onerror = reject;
        img.src = src;
      });
    }

    // Agrega una transición de deslizamiento a la imagen
    // @param {HTMLElement} container - Contenedor de la imagen
    // @param {string} src - URL de la imagen
    function slideUpImageTransition(container, src) {
      return new Promise(resolve => {
        const img = document.createElement('img');
        const size = container.clientHeight;
        img.src = src;
        img.width = size;
        img.height = size;
        img.style.top = '100%';
        container.appendChild(img);
        const lastImg = container.querySelector('img:last-child');
        setTimeout(() => {
          lastImg.style.top = 0;
        }, 100);
        lastImg.addEventListener('transitionend', () => {
          const allImgExcLast = container.querySelectorAll('img:not(:last-child)');
          allImgExcLast.forEach(img => img.remove());
          resolve(); // Resolvemos la promesa cuando termina la animación
        }, {
          once: true
        });
      });
    }
    const playerTitle = document.querySelector('.player-title');
    const playerDesc = document.querySelector('.player-desc');
    const imageChange = document.querySelectorAll('.image-change');
    const docTitle = document.title;
    function setStationInfo(data) {
      playerTitle.textContent = data.name;
      playerDesc.textContent = data.description;
      imageChange.forEach(el => {
        const img = el.querySelector('img:first-child');
        if (img) {
          img.src = data.image;
        }
      });
    }

    // Set current song data
    // @param {Object} data - Song data
    // @param {string} picture - URL of the picture
    function setCurrentSong(data, picture) {
      const {
        title,
        artist,
        artwork
      } = data;
      playerTitle.textContent = title;
      playerDesc.textContent = artist;
      observeTextScroll(playerTitle);
      document.title = `${title} - ${artist} | ${docTitle}`;
      const coverUrl = artwork?.xl || artwork?.large || artwork || picture;
      const $img = document.createElement('img');
      $img.src = coverUrl;
      imageChange.forEach(el => {
        loadImage(coverUrl).then(() => {
          slideUpImageTransition(el, coverUrl);
        }).catch(() => {
          console.error('Error loading image');
        });
      });
    }
    function convertSecondsToMinutes(seconds) {
      const minutes = Math.floor(seconds / 60);
      const remainingSeconds = seconds % 60;
      return `${minutes}:${remainingSeconds.toFixed(0) < 10 ? '0' : ''}${remainingSeconds.toFixed(0)}`;
    }

    // Obtener porcentaje entre dos valores
    function getPercentage(value, total) {
      return value * 100 / total;
    }
    function songProgress({
      el,
      duration,
      elapsed
    }) {
      const progress = getPercentage(elapsed, duration);

      // Agregar la variable css con el progreso de la canción
      el.style.setProperty('--song-progress', `${2 + progress}%`);

      // Actualizar el progreso de la canción
      const songElapsed = document.querySelector('.song-elapsed');
      const songDuration = document.querySelector('.song-duration');
      if (songElapsed) {
        songElapsed.textContent = convertSecondsToMinutes(elapsed);
      }
      if (songDuration) {
        songDuration.textContent = convertSecondsToMinutes(duration);
      }
    }

    /* eslint-disable camelcase */
    const stationOnAir = document.querySelector('.player-station-onair');
    const stationTitle = document.querySelector('.player-station-title');
    const stationImg = document.querySelector('.player-station-picture img');

    // Función para obtener el día actual en formato de 2 letras
    function getCurrentDay() {
      const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
      const date = new Date();
      const day = date.getDay();
      return days[day];
    }
    function compareNumbers(a, b) {
      return Number(a) === Number(b);
    }
    function sortByDay(programas) {
      const resultado = {
        monday: [],
        tuesday: [],
        wednesday: [],
        thursday: [],
        friday: [],
        saturday: [],
        sunday: []
      };
      programas.forEach(programa => {
        const {
          prog_horario_lu,
          prog_horario_ma,
          prog_horario_mi,
          prog_horario_ju,
          prog_horario_vi,
          prog_horario_sa,
          prog_horario_do,
          prog_lu,
          prog_ma,
          prog_mi,
          prog_ju,
          prog_vi,
          prog_sa,
          prog_do,
          prog_titulo,
          prog_descripcion,
          prog_foto
        } = programa;
        const item = {
          titulo: prog_titulo,
          descripcion: prog_descripcion,
          foto: prog_foto,
          horario: null
        };

        // Verificar cada día y agregar al arreglo correspondiente si está activo
        if (compareNumbers(prog_lu, 1)) {
          item.horario = prog_horario_lu;
          resultado.monday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ma, 1)) {
          item.horario = prog_horario_ma;
          resultado.tuesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_mi, 1)) {
          item.horario = prog_horario_mi;
          resultado.wednesday.push({
            ...item
          });
        }
        if (compareNumbers(prog_ju, 1)) {
          item.horario = prog_horario_ju;
          resultado.thursday.push({
            ...item
          });
        }
        if (compareNumbers(prog_vi, 1)) {
          item.horario = prog_horario_vi;
          resultado.friday.push({
            ...item
          });
        }
        if (compareNumbers(prog_sa, 1)) {
          item.horario = prog_horario_sa;
          resultado.saturday.push({
            ...item
          });
        }
        if (compareNumbers(prog_do, 1)) {
          item.horario = prog_horario_do;
          resultado.sunday.push({
            ...item
          });
        }
      });

      // Ordenar los programas por hora en cada día
      Object.keys(resultado).forEach(dia => {
        resultado[dia].sort((a, b) => a.horario.localeCompare(b.horario));
      });
      return resultado;
    }

    // Compara la hora actual con la hora de inicio de un programa
    // @param {string} time - Hora de inicio del programa
    // @returns {number} - 1 si la hora actual es mayor, -1 si la hora actual es menor, 0 si son iguales
    function compareTime(time) {
      const date = new Date();
      const currentHour = date.getHours();
      const currentMinutes = date.getMinutes();
      const [programHour, programMinutes] = time.split(':').map(Number);
      if (currentHour > programHour || currentHour === programHour && currentMinutes > programMinutes) {
        return 1; // Hora actual es mayor
      } else if (currentHour < programHour || currentHour === programHour && currentMinutes < programMinutes) {
        return -1; // Hora actual es menor
      } else {
        return 0; // Horas iguales
      }
    }
    function setupSchedule(data, baseUrl) {
      const programs = sortByDay(data);
      const currentDay = getCurrentDay();
      const currenDayPrograms = programs[currentDay];
      const currentProgram = currenDayPrograms.find((program, index, array) => {
        const nextProgram = array[index + 1];
        return compareTime(program.horario) === 1 && (!nextProgram || compareTime(nextProgram.horario) === -1);
      });
      if (currentProgram) {
        stationOnAir.innerHTML = `<div class="player-onair"><strong>EN VIVO</strong> Desde Las: ${currentProgram.horario}</div>`;
        stationTitle.textContent = currentProgram.titulo;
        stationImg.src = `${baseUrl}/${currentProgram.foto}`;
      }
    }

    // Función para inicializar el lienzo (canvas)
    function initCanvas(container) {
      const canvas = document.createElement('canvas');
      canvas.setAttribute('id', 'visualizerCanvas');
      canvas.setAttribute('class', 'visualizer-item');
      container.appendChild(canvas);
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
      return canvas;
    }

    // Función para cambiar el lienzo según el tamaño del contenedor
    function resizeCanvas(canvas, container) {
      canvas.width = container.clientWidth;
      canvas.height = container.clientHeight;
    }

    // Visualizer
    const visualizer = (audio, container) => {
      if (!audio || !container) {
        return;
      }
      const options = {
        fftSize: container.dataset.fftSize || 2048,
        numBars: container.dataset.bars || 80,
        maxHeight: 255
      };
      const ctx = new AudioContext();
      const audioSource = ctx.createMediaElementSource(audio);
      const analyzer = ctx.createAnalyser();
      audioSource.connect(analyzer);
      audioSource.connect(ctx.destination);
      const frequencyData = new Uint8Array(analyzer.frequencyBinCount);
      const canvas = initCanvas(container);
      const canvasCtx = canvas.getContext('2d');

      // Crear barras
      const renderBars = () => {
        resizeCanvas(canvas, container);
        analyzer.getByteFrequencyData(frequencyData);
        {
          analyzer.fftSize = options.fftSize;
        }
        canvasCtx.clearRect(0, 0, canvas.width, canvas.height);
        for (let i = 0; i < options.numBars; i++) {
          const index = Math.floor((i + 10) * (i < options.numBars / 2 ? 2 : 1));
          const fd = frequencyData[index];
          const barHeight = Math.max(4, fd / 255 * options.maxHeight);
          const barWidth = canvas.width / options.numBars;
          const x = i * barWidth;
          const y = canvas.height - barHeight;
          canvasCtx.fillStyle = 'white';
          canvasCtx.fillRect(x, y, barWidth - 2, barHeight);
        }
        requestAnimationFrame(renderBars);
      };
      renderBars();

      // Listener del cambio de espacio en la ventana
      window.addEventListener('resize', () => {
        resizeCanvas(canvas, container);
      });
    };
    const visualizerContainer = document.querySelector('.visualizer');
    const audio = new Audio();
    audio.crossOrigin = 'anonymous';
    const playerPlay = document.querySelector('.player-button-play');
    const svgIcons = {
      play: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M10 16.5v-9l6 4.5M12 2A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>',
      pause: '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24"><path d="M15 16h-2V8h2m-4 8H9V8h2m1-6A10 10 0 0 0 2 12a10 10 0 0 0 10 10 10 10 0 0 0 10-10A10 10 0 0 0 12 2Z"/></svg>'
    };
    let hasVisualizer = false;
    function play() {
      audio.load();
      audio.play();
      playerPlay.innerHTML = svgIcons.pause;
      playerPlay.classList.add('is-playing');
      playerPlay.setAttribute('aria-label', 'Pausar');
      playerPlay.setAttribute('aria-pressed', 'true');

      // Visualizer
      if (!hasVisualizer) {
        visualizer(audio, visualizerContainer);
        hasVisualizer = true;
      }
    }
    function pause() {
      audio.pause();
      playerPlay.innerHTML = svgIcons.play;
      playerPlay.classList.remove('is-playing');
      playerPlay.setAttribute('aria-label', 'Reproducir');
      playerPlay.setAttribute('aria-pressed', 'false');
    }
    function initAudio(url) {
      audio.src = url;
      playerPlay.addEventListener('click', () => {
        if (audio.paused) {
          play();
        } else {
          pause();
        }
      });
    }

    // Cargar datos de la canción actual al navegador
    function setMediaSession(data, picture) {
      if (!('mediaSession' in navigator)) {
        return;
      }
      const {
        artist,
        artwork
      } = data;
      const title = data.title || data.song;
      navigator.mediaSession.metadata = new MediaMetadata({
        title,
        artist,
        artwork: [{
          src: artwork || picture,
          sizes: '128x128',
          type: 'image/png'
        }, {
          src: artwork || picture,
          sizes: '256x256',
          type: 'image/png'
        }, {
          src: artwork || picture,
          sizes: '512x512',
          type: 'image/png'
        }]
      });
      navigator.mediaSession.setActionHandler('play', () => {
        play();
      });
      navigator.mediaSession.setActionHandler('pause', () => {
        pause();
      });
    }
    const progressElement = document.querySelector('.player-progress');
    let currentQuery = '';
    const api = `${config.baseUrl}api/`;
    const mainFetchUrl = `${api}radio-app/${config.idUser}`;
    const programFetchUrl = `${api}programa-app/${config.idUser}`;
    async function initPlayer() {
      const response = await fetch(mainFetchUrl);
      const data = await response.json();
      const stationData = normalizeData(data);
      setStationInfo(stationData);
      initAudio(stationData.stream);
      async function init() {
        const streamData = await getStreamData(stationData.stream);
        const query = createQuery(streamData);
        if (streamData.now_playing) {
          songProgress({
            el: progressElement,
            duration: streamData.now_playing.duration,
            elapsed: streamData.now_playing.elapsed
          });
        }
        if (query !== currentQuery) {
          const tempSongData = await getDataFromSearchApi(query, config.service);
          const songData = setDefaultData({
            data: tempSongData,
            song: streamData,
            stream: stationData
          });
          setCurrentSong(songData, stationData.cover);
          setMediaSession(songData, stationData.cover);
          currentQuery = query;
        }
        setTimeout(() => {
          init();
        }, config.timeRefresh);
      }
      init();
    }
    async function initProgram() {
      const response = await fetch(programFetchUrl);
      const data = await response.json();
      setupSchedule(data, config.baseUrl);
      setTimeout(() => {
        initProgram();
      }, config.programRefresh);
    }
    initPlayer();
    initProgram();
  }

  const players = {
    mainplayer,
    playerstatic,
    videoplayer,
    smartplayer
    // Agrega aquí otros reproductores si es necesario
  };
  installPWA();
  initPanel(players);

})();
